/**
 * Run All Migrations
 * Executes all database migrations in order
 */

require('dotenv').config();
const addCleaningStatusToBeds = require('./add-cleaning-status-to-beds');

const runAllMigrations = async () => {
    try {
        console.log('🚀 Starting migration process...\n');
        console.log('═══════════════════════════════════════════════════════');
        console.log('   CLINIC PRO - DATABASE MIGRATIONS');
        console.log('═══════════════════════════════════════════════════════\n');

        // List of all migrations
        const migrations = [
            {
                name: 'Add Cleaning Status to Beds',
                description: 'Adds Cleaning status to bed ENUM for automatic bed management',
                run: addCleaningStatusToBeds
            }
            // Add more migrations here as they are created
        ];

        console.log(`📋 Found ${migrations.length} migration(s) to run\n`);

        // Run each migration
        for (let i = 0; i < migrations.length; i++) {
            const migration = migrations[i];
            console.log(`\n${'─'.repeat(60)}`);
            console.log(`📦 Migration ${i + 1}/${migrations.length}: ${migration.name}`);
            console.log(`   ${migration.description}`);
            console.log(`${'─'.repeat(60)}\n`);

            try {
                await migration.run();
                console.log(`\n✅ Migration ${i + 1} completed successfully\n`);
            } catch (error) {
                console.error(`\n❌ Migration ${i + 1} failed: ${error.message}`);
                console.error('   Stopping migration process...\n');
                throw error;
            }
        }

        console.log('\n═══════════════════════════════════════════════════════');
        console.log('   ✨ ALL MIGRATIONS COMPLETED SUCCESSFULLY!');
        console.log('═══════════════════════════════════════════════════════\n');

        console.log('📊 Summary:');
        console.log(`   • Total migrations: ${migrations.length}`);
        console.log(`   • Successful: ${migrations.length}`);
        console.log(`   • Failed: 0\n`);

        console.log('🎉 Your database is now up to date!\n');
        console.log('Next steps:');
        console.log('   1. Restart your server to use new features');
        console.log('   2. Review CHANGELOG-INPATIENT-IMPROVEMENTS.md for details');
        console.log('   3. Test the new bed management features\n');

    } catch (error) {
        console.error('\n❌ MIGRATION PROCESS FAILED');
        console.error('   Please review the error above and try again.\n');
        console.error('Troubleshooting:');
        console.error('   1. Check database connection in .env file');
        console.error('   2. Ensure you have sufficient database permissions');
        console.error('   3. Verify required tables exist');
        console.error('   4. Review migrations/README.md for help\n');
        process.exit(1);
    }
};

// Run if called directly
if (require.main === module) {
    runAllMigrations().catch(console.error);
}

module.exports = runAllMigrations;

