/**
 * Standalone script to run the create-all-tables migration
 * Usage: node backend/migrations/run-create-all-tables.js
 */

require('dotenv').config({ path: require('path').join(__dirname, '../.env') });
const createAllTables = require('./002-create-all-tables-simple');
const fs = require('fs');
const path = require('path');

// Execute single statement with idempotent error handling
const executeStatement = async (sequelize, statement) => {
    try {
        await sequelize.query(statement);
    } catch (error) {
        const errorCode = error.original?.code || error.code;
        const errorMessage = error.original?.message || error.message || '';
        if (['ER_DUP_FIELDNAME', 'ER_DUP_KEYNAME', 'ER_CANT_DROP_FIELD_OR_KEY', 'ER_TABLE_EXISTS_ERROR', 'ER_DUP_ENTRY'].includes(errorCode) ||
            errorMessage.includes('already exists') ||
            errorMessage.includes('Duplicate') ||
            errorMessage.includes("doesn't exist") ||
            errorMessage.includes('Unknown table')) {
            return;
        }
        throw error;
    }
};

// Execute SQL file
const executeSqlFile = async (sequelize, filePath) => {
    let sql = await fs.promises.readFile(filePath, 'utf8');
    
    // Handle DELIMITER statements for stored procedures
    const delimiterRegex = /DELIMITER\s+(\$\$|;)/gi;
    let delimiter = ';';
    let statements = [];
    
    // Check if file uses DELIMITER
    if (delimiterRegex.test(sql)) {
        // Split by DELIMITER statements
        const parts = sql.split(/DELIMITER\s+(\$\$|;)/gi);
        
        for (let i = 0; i < parts.length; i++) {
            const part = parts[i].trim();
            
            // Skip DELIMITER command itself
            if (part.match(/^\$\$|^;$/i)) {
                delimiter = part === '$$' ? '$$' : ';';
                continue;
            }
            
            if (!part) continue;
            
            // Split by current delimiter
            const partStatements = part
                .split(delimiter)
                .map(s => s.trim())
                .filter(s => s.length > 0 && !s.startsWith('--') && !s.match(/^\/\*/));
            
            statements.push(...partStatements);
        }
    } else {
        // No DELIMITER, split by semicolons as usual
        statements = sql
            .split(';')
            .map(s => s.trim())
            .filter(s => s.length > 0 && !s.startsWith('--') && !s.match(/^\/\*/));
    }
    
    for (const statement of statements) {
        if (statement.trim() && !statement.match(/^DELIMITER/i)) {
            try {
                await sequelize.query(statement);
            } catch (error) {
                // Ignore duplicate column/key/table errors (idempotent)
                const errorCode = error.original?.code || error.code;
                const errorMessage = error.original?.message || error.message || '';
                
                // Check for various duplicate/ignore errors
                if (['ER_DUP_FIELDNAME', 'ER_DUP_KEYNAME', 'ER_CANT_DROP_FIELD_OR_KEY', 'ER_TABLE_EXISTS_ERROR', 'ER_DUP_ENTRY', 'ER_NO_SUCH_TABLE', 'ER_BAD_TABLE_ERROR'].includes(errorCode) ||
                    errorMessage.includes('Duplicate column') ||
                    errorMessage.includes('Duplicate key name') ||
                    errorMessage.includes('already exists') ||
                    errorMessage.includes("doesn't exist") ||
                    errorMessage.includes('Unknown table')) {
                    // Silently ignore - these are expected in some migrations
                    continue;
                }
                throw error;
            }
        }
    }
};

// Ensure critical new tables exist even if prior migrations were missed
const ensureExtraTables = async (sequelize) => {
    // tbl_laboratory_item (from 016)
    await executeStatement(sequelize, `
        CREATE TABLE IF NOT EXISTS tbl_laboratory_item (
            id INT NOT NULL AUTO_INCREMENT,
            barcode VARCHAR(50) NULL,
            item_name VARCHAR(50) NOT NULL,
            cat_id INT NULL,
            buy_price DECIMAL(10, 2) DEFAULT 0.00,
            sale_price DECIMAL(10, 2) DEFAULT 0.00,
            unit VARCHAR(50) NULL COMMENT 'Unit of measurement (e.g., mg/dL, mmol/L, g/L)',
            reference_ranges TEXT NULL COMMENT 'Normal reference ranges for the test',
            remark TEXT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            INDEX idx_barcode (barcode),
            INDEX idx_item_name (item_name),
            INDEX idx_cat_id (cat_id),
            INDEX idx_unit (unit),
            FOREIGN KEY (cat_id) REFERENCES tbl_service_catalog(id) ON DELETE SET NULL
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
    `);
    
    // Add unit and reference_ranges columns if they don't exist (from 021)
    try {
        const [unitCheck] = await sequelize.query(`
            SELECT COUNT(*) as count 
            FROM INFORMATION_SCHEMA.COLUMNS 
            WHERE TABLE_SCHEMA = DATABASE() 
            AND TABLE_NAME = 'tbl_laboratory_item' 
            AND COLUMN_NAME = 'unit'
        `);
        
        if (unitCheck[0].count === 0) {
            await executeStatement(sequelize, `
                ALTER TABLE tbl_laboratory_item 
                ADD COLUMN unit VARCHAR(50) NULL COMMENT 'Unit of measurement (e.g., mg/dL, mmol/L, g/L)' AFTER sale_price
            `);
            
            await executeStatement(sequelize, `
                ALTER TABLE tbl_laboratory_item 
                ADD COLUMN reference_ranges TEXT NULL COMMENT 'Normal reference ranges for the test' AFTER unit
            `);
            
            await executeStatement(sequelize, `
                CREATE INDEX idx_unit ON tbl_laboratory_item(unit)
            `);
        }
    } catch (error) {
        // Ignore errors - columns might already exist
    }

    // tbl_lab_order (from 017)
    await executeStatement(sequelize, `
        CREATE TABLE IF NOT EXISTS tbl_lab_order (
            id INT NOT NULL AUTO_INCREMENT,
            order_number VARCHAR(50) NOT NULL UNIQUE,
            sale_id INT NULL COMMENT 'Reference to tbl_sale when lab item was sold',
            sale_vno INT NULL COMMENT 'Voucher number from sale',
            lab_item_id INT NULL COMMENT 'Reference to tbl_laboratory_item',
            lab_item_name VARCHAR(255) NOT NULL COMMENT 'Lab item name at time of sale',
            patient_id VARCHAR(50) NOT NULL COMMENT 'Patient ID from sale',
            patient_name VARCHAR(255) NULL COMMENT 'Patient name',
            doctor_id INT NULL COMMENT 'Doctor ID from sale',
            doctor_name VARCHAR(255) NULL COMMENT 'Doctor name',
            referring_doctor_id INT NULL COMMENT 'Referring doctor ID',
            referring_doctor_name VARCHAR(255) NULL COMMENT 'Referring doctor name',
            order_date DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP COMMENT 'When lab item was sold/ordered',
            priority ENUM('Routine', 'Urgent', 'STAT', 'Emergency') DEFAULT 'Routine',
            clinical_info TEXT NULL COMMENT 'Clinical indication/reason for test',
            status ENUM('Ordered', 'Specimen_Collected', 'In_Progress', 'Completed', 'Cancelled') DEFAULT 'Ordered',
            collection_date DATETIME NULL,
            collected_by INT NULL COMMENT 'User ID of person who collected specimen',
            notes TEXT NULL,
            total_amount DECIMAL(10,2) DEFAULT 0.00,
            created_by INT NOT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            INDEX idx_order_number (order_number),
            INDEX idx_sale_id (sale_id),
            INDEX idx_sale_vno (sale_vno),
            INDEX idx_lab_item_id (lab_item_id),
            INDEX idx_patient_id (patient_id),
            INDEX idx_doctor_id (doctor_id),
            INDEX idx_status (status),
            INDEX idx_order_date (order_date),
            FOREIGN KEY (lab_item_id) REFERENCES tbl_laboratory_item(id) ON DELETE SET NULL,
            FOREIGN KEY (doctor_id) REFERENCES tbl_doctor(id) ON DELETE SET NULL
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
    `);

    // tbl_laboratory_package (from 020)
    await executeStatement(sequelize, `
        CREATE TABLE IF NOT EXISTS tbl_laboratory_package (
            id INT NOT NULL AUTO_INCREMENT,
            package_name VARCHAR(255) NOT NULL,
            package_code VARCHAR(50) NULL UNIQUE COMMENT 'Unique code for package (e.g., PKG001)',
            description TEXT NULL,
            total_price DECIMAL(10, 2) DEFAULT 0.00 COMMENT 'Total price of package (sum of all items)',
            is_active TINYINT(1) DEFAULT 1,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            INDEX idx_package_name (package_name),
            INDEX idx_package_code (package_code),
            INDEX idx_is_active (is_active)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
    `);

    // tbl_laboratory_package_item (from 020)
    await executeStatement(sequelize, `
        CREATE TABLE IF NOT EXISTS tbl_laboratory_package_item (
            id INT NOT NULL AUTO_INCREMENT,
            package_id INT NOT NULL,
            lab_item_id INT NOT NULL,
            quantity INT DEFAULT 1 COMMENT 'Quantity of this item in the package',
            item_price DECIMAL(10, 2) DEFAULT 0.00 COMMENT 'Price of this item at time of package creation',
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            INDEX idx_package_id (package_id),
            INDEX idx_lab_item_id (lab_item_id),
            FOREIGN KEY (package_id) REFERENCES tbl_laboratory_package(id) ON DELETE CASCADE,
            FOREIGN KEY (lab_item_id) REFERENCES tbl_laboratory_item(id) ON DELETE CASCADE,
            UNIQUE KEY unique_package_item (package_id, lab_item_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
    `);

    // tbl_delete_history (from 018)
    await executeStatement(sequelize, `
        CREATE TABLE IF NOT EXISTS tbl_delete_history (
            id BIGINT NOT NULL AUTO_INCREMENT,
            table_name VARCHAR(100) NOT NULL,
            record_id VARCHAR(100) NULL,
            deleted_by INT NULL,
            username VARCHAR(100) NULL,
            old_data LONGTEXT NULL COMMENT 'JSON data stored as text',
            reason VARCHAR(255) NULL,
            ip_address VARCHAR(45) NULL,
            user_agent VARCHAR(255) NULL,
            deleted_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            INDEX idx_table (table_name),
            INDEX idx_deleted_by (deleted_by),
            INDEX idx_deleted_at (deleted_at)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
    `);
    
    // Fix old_data column if it was created with JSON type (for MariaDB compatibility)
    try {
        const [columnCheck] = await sequelize.query(`
            SELECT DATA_TYPE 
            FROM INFORMATION_SCHEMA.COLUMNS 
            WHERE TABLE_SCHEMA = DATABASE() 
            AND TABLE_NAME = 'tbl_delete_history' 
            AND COLUMN_NAME = 'old_data'
        `);
        
        if (columnCheck.length > 0 && columnCheck[0].DATA_TYPE === 'json') {
            await executeStatement(sequelize, `
                ALTER TABLE tbl_delete_history 
                MODIFY COLUMN old_data LONGTEXT NULL COMMENT 'JSON data stored as text'
            `);
        }
    } catch (error) {
        // Ignore errors - column might not exist or already correct
    }

    // tbl_notification_read - Track which notifications users have read
    await executeStatement(sequelize, `
        CREATE TABLE IF NOT EXISTS tbl_notification_read (
            id INT NOT NULL AUTO_INCREMENT,
            notification_id VARCHAR(191) NOT NULL COMMENT 'Max 191 for utf8mb4 index compatibility',
            user_id INT NULL,
            read_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY unique_notification_user (notification_id, user_id),
            INDEX idx_user_id (user_id),
            INDEX idx_notification_id (notification_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
    `);

    // Add supplier_id to tbl_stock (from 019)
    // Check if column exists first
    const [columnCheck] = await sequelize.query(`
        SELECT COUNT(*) as count 
        FROM INFORMATION_SCHEMA.COLUMNS 
        WHERE TABLE_SCHEMA = DATABASE() 
        AND TABLE_NAME = 'tbl_stock' 
        AND COLUMN_NAME = 'supplier_id'
    `);
    
    if (columnCheck[0].count === 0) {
        await executeStatement(sequelize, `
            ALTER TABLE tbl_stock
            ADD COLUMN supplier_id INT NULL AFTER cat_id_2
        `);
        
        // Add foreign key constraint (only if it doesn't exist)
        await executeStatement(sequelize, `
            ALTER TABLE tbl_stock
            ADD CONSTRAINT fk_stock_supplier 
            FOREIGN KEY (supplier_id) REFERENCES tbl_supplier(id) 
            ON DELETE SET NULL
        `);
        
        // Create index
        await executeStatement(sequelize, `
            CREATE INDEX idx_stock_supplier ON tbl_stock(supplier_id)
        `);
    }
    
    // Add supplier_id to tbl_purchase
    // Check if column exists first
    const [purchaseSupplierCheck] = await sequelize.query(`
        SELECT COUNT(*) as count 
        FROM INFORMATION_SCHEMA.COLUMNS 
        WHERE TABLE_SCHEMA = DATABASE() 
        AND TABLE_NAME = 'tbl_purchase' 
        AND COLUMN_NAME = 'supplier_id'
    `);
    
    if (purchaseSupplierCheck[0].count === 0) {
        await executeStatement(sequelize, `
            ALTER TABLE tbl_purchase
            ADD COLUMN supplier_id INT NULL AFTER purchase_voucher_id
        `);
        
        // Add foreign key constraint (only if it doesn't exist)
        await executeStatement(sequelize, `
            ALTER TABLE tbl_purchase
            ADD CONSTRAINT fk_purchase_supplier 
            FOREIGN KEY (supplier_id) REFERENCES tbl_supplier(id) 
            ON DELETE SET NULL
            ON UPDATE CASCADE
        `);
        
        // Create index
        await executeStatement(sequelize, `
            CREATE INDEX idx_purchase_supplier ON tbl_purchase(supplier_id)
        `);
    }

    // Add warehouse_id to tbl_sale (for sale return tracking)
    // Check if column exists first
    const [warehouseColumnCheck] = await sequelize.query(`
        SELECT COUNT(*) as count 
        FROM INFORMATION_SCHEMA.COLUMNS 
        WHERE TABLE_SCHEMA = DATABASE() 
        AND TABLE_NAME = 'tbl_sale' 
        AND COLUMN_NAME = 'warehouse_id'
    `);
    
    if (warehouseColumnCheck[0].count === 0) {
        // Check if referring_doctor_name column exists to determine where to add warehouse_id
        const [refDoctorCheck] = await sequelize.query(`
            SELECT COUNT(*) as count 
            FROM INFORMATION_SCHEMA.COLUMNS 
            WHERE TABLE_SCHEMA = DATABASE() 
            AND TABLE_NAME = 'tbl_sale' 
            AND COLUMN_NAME = 'referring_doctor_name'
        `);
        
        const afterColumn = refDoctorCheck[0].count > 0 ? 'referring_doctor_name' : 'cost_price';
        
        await executeStatement(sequelize, `
            ALTER TABLE tbl_sale
            ADD COLUMN warehouse_id INT NULL AFTER ${afterColumn}
        `);
        
        // Create index
        await executeStatement(sequelize, `
            CREATE INDEX idx_sale_warehouse_id ON tbl_sale(warehouse_id)
        `);
        
        // Add foreign key constraint (only if tbl_warehouse exists)
        try {
            const [warehouseTableCheck] = await sequelize.query(`
                SELECT COUNT(*) as count 
                FROM INFORMATION_SCHEMA.TABLES 
                WHERE TABLE_SCHEMA = DATABASE() 
                AND TABLE_NAME = 'tbl_warehouse'
            `);
            
            if (warehouseTableCheck[0].count > 0) {
                // Check if constraint already exists
                const [constraintCheck] = await sequelize.query(`
                    SELECT COUNT(*) as count 
                    FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS 
                    WHERE CONSTRAINT_SCHEMA = DATABASE() 
                    AND CONSTRAINT_NAME = 'fk_sale_warehouse' 
                    AND TABLE_NAME = 'tbl_sale'
                `);
                
                if (constraintCheck[0].count === 0) {
                    await executeStatement(sequelize, `
                        ALTER TABLE tbl_sale
                        ADD CONSTRAINT fk_sale_warehouse 
                        FOREIGN KEY (warehouse_id) REFERENCES tbl_warehouse(id) 
                        ON DELETE SET NULL 
                        ON UPDATE CASCADE
                    `);
                    console.log('   ✅ Added foreign key constraint fk_sale_warehouse');
                }
            } else {
                console.log('   ⚠️  tbl_warehouse table not found, skipping foreign key constraint');
            }
        } catch (error) {
            // Ignore constraint errors (might already exist)
            if (!error.message.includes('Duplicate key name') && !error.message.includes('already exists')) {
                console.log(`   ⚠️  Could not add foreign key constraint: ${error.message.substring(0, 50)}`);
            }
        }
        
        console.log('   ✅ Added warehouse_id column to tbl_sale');
    }
    
    // Add warehouse_id to tbl_stock_expense
    // Check if column exists first
    const [stockExpenseWarehouseCheck] = await sequelize.query(`
        SELECT COUNT(*) as count 
        FROM INFORMATION_SCHEMA.COLUMNS 
        WHERE TABLE_SCHEMA = DATABASE() 
        AND TABLE_NAME = 'tbl_stock_expense' 
        AND COLUMN_NAME = 'warehouse_id'
    `);
    
    if (stockExpenseWarehouseCheck[0].count === 0) {
        await executeStatement(sequelize, `
            ALTER TABLE tbl_stock_expense
            ADD COLUMN warehouse_id INT NULL AFTER date
        `);
        
        // Create index
        await executeStatement(sequelize, `
            CREATE INDEX idx_stock_expense_warehouse ON tbl_stock_expense(warehouse_id)
        `);
        
        // Add foreign key constraint (only if tbl_warehouse exists)
        try {
            const [warehouseTableCheck] = await sequelize.query(`
                SELECT COUNT(*) as count 
                FROM INFORMATION_SCHEMA.TABLES 
                WHERE TABLE_SCHEMA = DATABASE() 
                AND TABLE_NAME = 'tbl_warehouse'
            `);
            
            if (warehouseTableCheck[0].count > 0) {
                // Check if constraint already exists
                const [constraintCheck] = await sequelize.query(`
                    SELECT COUNT(*) as count 
                    FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS 
                    WHERE CONSTRAINT_SCHEMA = DATABASE() 
                    AND CONSTRAINT_NAME = 'fk_stock_expense_warehouse' 
                    AND TABLE_NAME = 'tbl_stock_expense'
                `);
                
                if (constraintCheck[0].count === 0) {
                    await executeStatement(sequelize, `
                        ALTER TABLE tbl_stock_expense
                        ADD CONSTRAINT fk_stock_expense_warehouse 
                        FOREIGN KEY (warehouse_id) REFERENCES tbl_warehouse(id) 
                        ON DELETE SET NULL 
                        ON UPDATE CASCADE
                    `);
                    console.log('   ✅ Added foreign key constraint fk_stock_expense_warehouse');
                }
            } else {
                console.log('   ⚠️  tbl_warehouse table not found, skipping foreign key constraint');
            }
        } catch (error) {
            // Ignore constraint errors (might already exist)
            if (!error.message.includes('Duplicate key name') && !error.message.includes('already exists')) {
                console.log(`   ⚠️  Could not add foreign key constraint: ${error.message.substring(0, 50)}`);
            }
        }
        
        console.log('   ✅ Added warehouse_id column to tbl_stock_expense');
    }

    // Add payment_method to tbl_expenses
    const [expensePaymentMethodCheck] = await sequelize.query(`
        SELECT COUNT(*) as count
        FROM INFORMATION_SCHEMA.COLUMNS
        WHERE TABLE_SCHEMA = DATABASE()
        AND TABLE_NAME = 'tbl_expenses'
        AND COLUMN_NAME = 'payment_method'
    `);

    if (expensePaymentMethodCheck[0].count === 0) {
        await executeStatement(sequelize, `
            ALTER TABLE tbl_expenses
            ADD COLUMN payment_method VARCHAR(50) NULL AFTER remark
        `);
        console.log('   ✅ Added payment_method column to tbl_expenses');
    }
    
    // Ensure warehouse stock trigger exists
    try {
        // Drop trigger if it exists (to recreate it)
        await sequelize.query('DROP TRIGGER IF EXISTS trg_update_warehouse_stock_on_movement');
        
        // Create the trigger
        await sequelize.query(`
            CREATE TRIGGER trg_update_warehouse_stock_on_movement
            AFTER INSERT ON tbl_stock_movement
            FOR EACH ROW
            BEGIN
                IF NEW.movement_type IN ('IN', 'TRANSFER_IN') THEN
                    INSERT INTO tbl_warehouse_stock (warehouse_id, stock_id, quantity, updated_by)
                    VALUES (NEW.warehouse_id, NEW.stock_id, NEW.quantity, NEW.created_by)
                    ON DUPLICATE KEY UPDATE 
                        quantity = quantity + NEW.quantity,
                        updated_by = NEW.created_by,
                        last_updated = CURRENT_TIMESTAMP;
                ELSEIF NEW.movement_type IN ('OUT', 'TRANSFER_OUT') THEN
                    UPDATE tbl_warehouse_stock
                    SET quantity = GREATEST(0, quantity - NEW.quantity),
                        updated_by = NEW.created_by,
                        last_updated = CURRENT_TIMESTAMP
                    WHERE warehouse_id = NEW.warehouse_id AND stock_id = NEW.stock_id;
                END IF;
            END
        `);
        console.log('   ✅ Warehouse stock trigger created/updated');
    } catch (error) {
        // Ignore if trigger already exists or table doesn't exist yet
        if (!error.message.includes('already exists') && !error.message.includes("doesn't exist")) {
            console.warn('   ⚠️  Could not create warehouse stock trigger:', error.message);
        }
    }
};

const runMigration = async () => {
    try {
        console.log('🚀 Starting Create All Tables Migration...\n');
        console.log('═══════════════════════════════════════════════════════');
        console.log('   CLINIC PRO - CREATE ALL 97 TABLES & TRIGGER');
        console.log('═══════════════════════════════════════════════════════\n');

        // Import Sequelize and create queryInterface
        const { Sequelize } = require('sequelize');
        const sequelize = new Sequelize(
            process.env.DB_NAME || 'clinic_pro_db',
            process.env.DB_USER || 'root',
            process.env.DB_PASSWORD || '',
            {
                host: process.env.DB_HOST || '127.0.0.1',
                port: parseInt(process.env.DB_PORT) || 3306,
                dialect: 'mysql',
                logging: false,
                pool: {
                    max: 5,
                    min: 0,
                    acquire: 30000,
                    idle: 10000
                }
            }
        );

        // Create a mock queryInterface
        const queryInterface = sequelize.getQueryInterface();

        // Step 1: Run the main migration (creates 97 tables)
        console.log('📋 Step 1: Creating initial 97 tables and trigger...');
        await createAllTables.up(queryInterface, Sequelize);
        console.log('✅ Initial tables created\n');

        // Step 2: Run SQL migration files from database/migrations/
        const migrationsDir = path.join(__dirname, '../../database/migrations');
        if (fs.existsSync(migrationsDir)) {
            console.log('📋 Step 2: Running additional SQL migrations...');
            const files = await fs.promises.readdir(migrationsDir);
            
            // Filter and prioritize cloud versions if they exist
            const migrationMap = new Map();
            const cloudFiles = new Set();
            
            files
                .filter(f => f.endsWith('.sql') && f !== 'warehouse_system.sql')
                .forEach(f => {
                    if (f.includes('_cloud.sql')) {
                        const baseName = f.replace('_cloud.sql', '');
                        migrationMap.set(baseName, f);
                        cloudFiles.add(baseName);
                    }
                });
            
            // Add regular files, but skip if cloud version exists
            files
                .filter(f => f.endsWith('.sql') && f !== 'warehouse_system.sql' && !f.includes('_cloud.sql'))
                .forEach(f => {
                    const baseName = f.replace('.sql', '');
                    if (!cloudFiles.has(baseName)) {
                        migrationMap.set(baseName, f);
                    }
                });
            
            const migrationFiles = Array.from(migrationMap.values())
                .sort(); // Sort to ensure order (001, 002, 016, 017, etc.)
            
            console.log(`   Found ${migrationFiles.length} migration files\n`);
            
            for (const file of migrationFiles) {
                const filePath = path.join(migrationsDir, file);
                console.log(`   🔄 Running ${file}...`);
                try {
                    await executeSqlFile(sequelize, filePath);
                    console.log(`   ✅ Completed ${file}\n`);
                } catch (error) {
                    console.error(`   ❌ Error in ${file}:`, error.message);
                    throw error;
                }
            }
            console.log('✅ All SQL migrations completed\n');
        } else {
            console.log('⚠️  Migrations directory not found, skipping SQL migrations\n');
        }

        // Step 3: Ensure critical newer tables exist and run critical migrations
        console.log('📋 Step 3: Ensuring critical tables and migrations (lab items, lab orders, lab packages, delete history, notification read, supplier link, warehouse links)...');
        await ensureExtraTables(sequelize);
        console.log('✅ Critical tables and migrations ensured\n');

        console.log('═══════════════════════════════════════════════════════');
        console.log('   ✨ MIGRATION COMPLETED SUCCESSFULLY!');
        console.log('═══════════════════════════════════════════════════════\n');

        await sequelize.close();
        process.exit(0);

    } catch (error) {
        console.error('\n❌ MIGRATION FAILED');
        console.error('   Error:', error.message);
        console.error('   Stack:', error.stack);
        console.error('\nTroubleshooting:');
        console.error('   1. Check database connection in .env file');
        console.error('   2. Ensure clinic_pro_db.sql exists in project root');
        console.error('   3. Verify you have sufficient database permissions');
        console.error('   4. Check if tables already exist (migration is idempotent)\n');
        process.exit(1);
    }
};

// Run if called directly
if (require.main === module) {
    runMigration();
}

module.exports = runMigration;

