/**
 * Verify Database Setup
 * Checks if all required tables and columns exist with correct structure
 */

require('dotenv').config();
const { pool } = require('../config/database');

const verifySetup = async () => {
    try {
        console.log('🔍 Verifying database setup...\n');
        
        const checks = [];
        let allPassed = true;

        // Check 1: tbl_beds table exists
        console.log('1️⃣  Checking if tbl_beds table exists...');
        const [tables] = await pool.execute(`
            SELECT TABLE_NAME 
            FROM INFORMATION_SCHEMA.TABLES 
            WHERE TABLE_SCHEMA = DATABASE() 
            AND TABLE_NAME = 'tbl_beds'
        `);
        
        if (tables.length > 0) {
            console.log('   ✅ tbl_beds table exists');
            checks.push({ check: 'tbl_beds table', status: 'passed' });
        } else {
            console.log('   ❌ tbl_beds table not found');
            checks.push({ check: 'tbl_beds table', status: 'failed' });
            allPassed = false;
        }

        // Check 2: status column exists
        console.log('\n2️⃣  Checking if status column exists...');
        const [statusColumn] = await pool.execute(`
            SELECT COLUMN_NAME 
            FROM INFORMATION_SCHEMA.COLUMNS 
            WHERE TABLE_SCHEMA = DATABASE() 
            AND TABLE_NAME = 'tbl_beds' 
            AND COLUMN_NAME = 'status'
        `);
        
        if (statusColumn.length > 0) {
            console.log('   ✅ status column exists');
            checks.push({ check: 'status column', status: 'passed' });
        } else {
            console.log('   ❌ status column not found');
            checks.push({ check: 'status column', status: 'failed' });
            allPassed = false;
        }

        // Check 3: Cleaning status in ENUM
        console.log('\n3️⃣  Checking if Cleaning status is in ENUM...');
        const [enumValues] = await pool.execute(`
            SELECT COLUMN_TYPE 
            FROM INFORMATION_SCHEMA.COLUMNS 
            WHERE TABLE_SCHEMA = DATABASE() 
            AND TABLE_NAME = 'tbl_beds' 
            AND COLUMN_NAME = 'status'
        `);
        
        if (enumValues.length > 0) {
            const enumType = enumValues[0].COLUMN_TYPE;
            console.log(`   📊 Current ENUM: ${enumType}`);
            
            if (enumType.includes('Cleaning')) {
                console.log('   ✅ Cleaning status exists in ENUM');
                checks.push({ check: 'Cleaning status', status: 'passed' });
            } else {
                console.log('   ❌ Cleaning status not found in ENUM');
                console.log('   💡 Run: node migrations/add-cleaning-status-to-beds.js');
                checks.push({ check: 'Cleaning status', status: 'failed' });
                allPassed = false;
            }
        }

        // Check 4: tbl_inpatient table exists
        console.log('\n4️⃣  Checking if tbl_inpatient table exists...');
        const [inpatientTable] = await pool.execute(`
            SELECT TABLE_NAME 
            FROM INFORMATION_SCHEMA.TABLES 
            WHERE TABLE_SCHEMA = DATABASE() 
            AND TABLE_NAME = 'tbl_inpatient'
        `);
        
        if (inpatientTable.length > 0) {
            console.log('   ✅ tbl_inpatient table exists');
            checks.push({ check: 'tbl_inpatient table', status: 'passed' });
        } else {
            console.log('   ❌ tbl_inpatient table not found');
            checks.push({ check: 'tbl_inpatient table', status: 'failed' });
            allPassed = false;
        }

        // Check 5: Get current bed statistics
        console.log('\n5️⃣  Checking bed statistics...');
        const [bedStats] = await pool.execute(`
            SELECT status, COUNT(*) as count 
            FROM tbl_beds 
            GROUP BY status
        `);
        
        if (bedStats.length > 0) {
            console.log('   ✅ Bed statistics:');
            bedStats.forEach(stat => {
                console.log(`      • ${stat.status}: ${stat.count} bed(s)`);
            });
            checks.push({ check: 'Bed statistics', status: 'passed' });
        } else {
            console.log('   ⚠️  No beds found in database');
            console.log('   💡 Run: node seed-hospital-fixed.js');
            checks.push({ check: 'Bed statistics', status: 'warning' });
        }

        // Check 6: Check departments
        console.log('\n6️⃣  Checking departments...');
        const [deptStats] = await pool.execute(`
            SELECT COUNT(*) as count FROM tbl_departments
        `);
        
        if (deptStats[0].count > 0) {
            console.log(`   ✅ Found ${deptStats[0].count} department(s)`);
            checks.push({ check: 'Departments', status: 'passed' });
        } else {
            console.log('   ⚠️  No departments found');
            console.log('   💡 Run: node seed-hospital-fixed.js');
            checks.push({ check: 'Departments', status: 'warning' });
        }

        // Summary
        console.log('\n═══════════════════════════════════════════════════════');
        console.log('   VERIFICATION SUMMARY');
        console.log('═══════════════════════════════════════════════════════\n');

        const passed = checks.filter(c => c.status === 'passed').length;
        const failed = checks.filter(c => c.status === 'failed').length;
        const warnings = checks.filter(c => c.status === 'warning').length;

        console.log(`Total Checks: ${checks.length}`);
        console.log(`✅ Passed: ${passed}`);
        console.log(`❌ Failed: ${failed}`);
        console.log(`⚠️  Warnings: ${warnings}\n`);

        if (allPassed && failed === 0) {
            console.log('🎉 DATABASE SETUP IS COMPLETE!\n');
            console.log('Your system is ready to use with the following features:');
            console.log('   • Automatic bed cleaning after discharge');
            console.log('   • Enhanced bed availability warnings');
            console.log('   • Duplicate patient admission prevention');
            console.log('   • Manual bed cleaning endpoint\n');
        } else {
            console.log('⚠️  SETUP INCOMPLETE\n');
            console.log('Please review the failed checks above and:');
            console.log('   1. Run missing migrations');
            console.log('   2. Create missing tables');
            console.log('   3. Seed required data\n');
            console.log('For help, see: migrations/README.md\n');
        }

        // Detailed recommendations
        if (failed > 0 || warnings > 0) {
            console.log('📋 RECOMMENDED ACTIONS:\n');
            
            if (checks.some(c => c.check === 'tbl_beds table' && c.status === 'failed')) {
                console.log('   1. Create tables:');
                console.log('      cd backend/utils');
                console.log('      node create-all-missing-tables.js\n');
            }
            
            if (checks.some(c => c.check === 'Cleaning status' && c.status === 'failed')) {
                console.log('   2. Add Cleaning status:');
                console.log('      cd backend/migrations');
                console.log('      node add-cleaning-status-to-beds.js\n');
            }
            
            if (checks.some(c => c.status === 'warning')) {
                console.log('   3. Seed sample data:');
                console.log('      cd backend');
                console.log('      node seed-hospital-fixed.js\n');
            }
        }

    } catch (error) {
        console.error('❌ Verification failed:', error.message);
        console.error('\nTroubleshooting:');
        console.error('   1. Check database connection in .env file');
        console.error('   2. Ensure database exists');
        console.error('   3. Verify database credentials\n');
    } finally {
        await pool.end();
    }
};

// Run if called directly
if (require.main === module) {
    verifySetup().catch(console.error);
}

module.exports = verifySetup;

