/**
 * Base Model Class
 * Provides generic CRUD operations for all tables
 */

const { query, transaction } = require('../config/database');
const { recordDelete } = require('../utils/delete-history');

class BaseModel {
    constructor(tableName, primaryKey = 'id') {
        this.tableName = tableName;
        this.primaryKey = primaryKey;
    }

    /**
     * Get all records with pagination
     */
    async findAll(options = {}) {
        const {
            page = 1,
            limit = 20,
            orderBy = this.primaryKey,
            order = 'DESC',
            where = {},
            select = '*'
        } = options;

        const pageNum = parseInt(page) || 1;
        const limitNum = parseInt(limit) || 20;
        const offsetNum = (pageNum - 1) * limitNum;
        let sql = `SELECT ${select} FROM ${this.tableName}`;
        const whereParams = [];

        // Build WHERE clause
        const whereClauses = Object.keys(where).map(key => {
            whereParams.push(where[key]);
            return `${key} = ?`;
        });

        if (whereClauses.length > 0) {
            sql += ' WHERE ' + whereClauses.join(' AND ');
        }

        // Add ORDER BY and LIMIT (inline numeric values to avoid ER_WRONG_ARGUMENTS)
        sql += ` ORDER BY ${orderBy} ${order} LIMIT ${limitNum} OFFSET ${offsetNum}`;

        const results = await query(sql, whereParams);

        // Get total count
        let countSql = `SELECT COUNT(*) as total FROM ${this.tableName}`;
        if (whereClauses.length > 0) {
            countSql += ' WHERE ' + whereClauses.join(' AND ');
        }
        const countResult = await query(countSql, whereParams);

        return {
            data: results,
            pagination: {
                page: pageNum,
                limit: limitNum,
                total: countResult[0].total,
                pages: Math.ceil(countResult[0].total / limitNum)
            }
        };
    }

    /**
     * Find record by ID
     */
    async findById(id) {
        const sql = `SELECT * FROM ${this.tableName} WHERE ${this.primaryKey} = ?`;
        const results = await query(sql, [id]);
        return results[0] || null;
    }

    /**
     * Find one record by condition
     */
    async findOne(where = {}) {
        let sql = `SELECT * FROM ${this.tableName}`;
        const params = [];

        const whereClauses = Object.keys(where).map(key => {
            params.push(where[key]);
            return `${key} = ?`;
        });

        if (whereClauses.length > 0) {
            sql += ' WHERE ' + whereClauses.join(' AND ');
        }

        sql += ' LIMIT 1';

        const results = await query(sql, params);
        return results[0] || null;
    }

    /**
     * Create new record
     */
    async create(data) {
        const columns = Object.keys(data);
        const values = Object.values(data);
        const placeholders = columns.map(() => '?').join(', ');

        const sql = `INSERT INTO ${this.tableName} (${columns.join(', ')}) VALUES (${placeholders})`;
        const result = await query(sql, values);

        return {
            [this.primaryKey]: result.insertId,
            ...data
        };
    }

    /**
     * Update record by ID
     */
    async update(id, data) {
        const columns = Object.keys(data);
        const values = Object.values(data);
        const setClause = columns.map(col => `${col} = ?`).join(', ');

        const sql = `UPDATE ${this.tableName} SET ${setClause} WHERE ${this.primaryKey} = ?`;
        await query(sql, [...values, id]);

        return this.findById(id);
    }

    /**
     * Delete record by ID
     */
    async delete(id, options = {}) {
        // options may include { userId, username, reason, ip, userAgent }
        let oldRow = null;
        try {
            const existing = await query(
                `SELECT * FROM ${this.tableName} WHERE ${this.primaryKey} = ? LIMIT 1`,
                [id]
            );
            oldRow = existing && existing.length ? existing[0] : null;
        } catch (e) {
            // ignore fetch errors and continue with deletion
        }

        const sql = `DELETE FROM ${this.tableName} WHERE ${this.primaryKey} = ?`;
        const result = await query(sql, [id]);
        const success = result.affectedRows > 0;

        if (success) {
            await recordDelete({
                tableName: this.tableName,
                recordId: id,
                oldData: oldRow,
                deletedBy: options.userId ?? null,
                username: options.username ?? null,
                reason: options.reason ?? null,
                ip: options.ip ?? null,
                userAgent: options.userAgent ?? null,
            });
        }

        return success;
    }

    /**
     * Bulk insert
     */
    async bulkCreate(dataArray) {
        if (!dataArray || dataArray.length === 0) return [];

        const columns = Object.keys(dataArray[0]);
        const placeholders = dataArray.map(() => 
            `(${columns.map(() => '?').join(', ')})`
        ).join(', ');

        const values = dataArray.flatMap(data => Object.values(data));

        const sql = `INSERT INTO ${this.tableName} (${columns.join(', ')}) VALUES ${placeholders}`;
        await query(sql, values);

        return dataArray;
    }

    /**
     * Search records
     */
    async search(searchColumn, searchTerm, options = {}) {
        const { limit = 50 } = options;
        const sql = `SELECT * FROM ${this.tableName} WHERE ${searchColumn} LIKE ? LIMIT ?`;
        return await query(sql, [`%${searchTerm}%`, limit]);
    }

    /**
     * Count records
     */
    async count(where = {}) {
        let sql = `SELECT COUNT(*) as count FROM ${this.tableName}`;
        const params = [];

        const whereClauses = Object.keys(where).map(key => {
            params.push(where[key]);
            return `${key} = ?`;
        });

        if (whereClauses.length > 0) {
            sql += ' WHERE ' + whereClauses.join(' AND ');
        }

        const result = await query(sql, params);
        return result[0].count;
    }

    /**
     * Execute raw query
     */
    async raw(sql, params = []) {
        return await query(sql, params);
    }
}

module.exports = BaseModel;

