/**
 * Customer Model
 * Handles field mapping between frontend and database for customer data
 */

const BaseModel = require('./BaseModel');

class CustomerModel extends BaseModel {
    constructor() {
        super('tbl_customer', 'id');
    }

    /**
     * Map frontend fields to database columns
     */
    mapToDatabase(data) {
        const mapped = {};
        
        if (data.name !== undefined) mapped.name = data.name;
        if (data.phone !== undefined) mapped.ph_no = data.phone; // Map phone to ph_no
        if (data.email !== undefined) mapped.remark = data.email; // Store email in remark for now
        if (data.address !== undefined) mapped.address = data.address;
        if (data.type !== undefined) mapped.shop = data.type; // Store type in shop field
        if (data.date !== undefined) mapped.date = data.date;
        if (data.nrc_no !== undefined) mapped.nrc_no = data.nrc_no;
        
        return mapped;
    }

    /**
     * Map database columns to frontend fields
     */
    mapFromDatabase(record) {
        if (!record) return null;
        
        return {
            id: record.id,
            name: record.name,
            phone: record.ph_no, // Map ph_no to phone
            email: record.remark, // Map remark to email
            address: record.address,
            type: record.shop || 'regular', // Map shop to type, default to 'regular'
            created_date: record.date,
            nrc_no: record.nrc_no
        };
    }

    /**
     * Get all records with field mapping
     */
    async findAll(options = {}) {
        const result = await super.findAll(options);
        return {
            ...result,
            data: result.data.map(record => this.mapFromDatabase(record))
        };
    }

    /**
     * Find by ID with field mapping
     */
    async findById(id) {
        const record = await super.findById(id);
        return this.mapFromDatabase(record);
    }

    /**
     * Find one with field mapping
     */
    async findOne(where = {}) {
        const record = await super.findOne(where);
        return this.mapFromDatabase(record);
    }

    /**
     * Create with field mapping
     */
    async create(data) {
        const mappedData = this.mapToDatabase(data);
        
        // Add current date if not provided
        if (!mappedData.date) {
            mappedData.date = new Date().toISOString().split('T')[0];
        }
        
        const result = await super.create(mappedData);
        return this.mapFromDatabase(result);
    }

    /**
     * Update with field mapping
     */
    async update(id, data) {
        const mappedData = this.mapToDatabase(data);
        const result = await super.update(id, mappedData);
        return this.mapFromDatabase(result);
    }

    /**
     * Search customers
     */
    async search(searchTerm, options = {}) {
        const { limit = 50 } = options;
        const sql = `
            SELECT * FROM ${this.tableName} 
            WHERE name LIKE ? OR ph_no LIKE ? OR address LIKE ?
            LIMIT ?
        `;
        const results = await this.raw(sql, [`%${searchTerm}%`, `%${searchTerm}%`, `%${searchTerm}%`, limit]);
        return results.map(record => this.mapFromDatabase(record));
    }
}

module.exports = CustomerModel;

