const express = require('express');
const router = express.Router();
const { protect } = require('../middleware/auth');
const { requireCrudPermission } = require('../middleware/permissions');
const { query } = require('../config/database');

router.use(protect);
router.use(requireCrudPermission('appointments'));

// Get all appointments
router.get('/', async (req, res, next) => {
    try {
        const { limit } = req.query;
        const lim = Math.min(parseInt(limit, 10) || 1000, 1000);
        // Ensure table exists (self-healing)
        await query(`CREATE TABLE IF NOT EXISTS tbl_appointment (
            id INT NOT NULL AUTO_INCREMENT,
            name VARCHAR(50) NULL,
            phone VARCHAR(50) NULL,
            case_ VARCHAR(50) NULL,
            dr_id INT NULL,
            dr_name VARCHAR(50) NULL,
            token_number VARCHAR(50) NULL,
            time DATETIME NULL,
            status VARCHAR(20) DEFAULT 'scheduled',
            PRIMARY KEY (id),
            INDEX idx_dr_id (dr_id),
            INDEX idx_status (status),
            INDEX idx_time_status (time, status)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`);
        
        // Add dr_id column if it doesn't exist
        try {
            await query(`ALTER TABLE tbl_appointment ADD COLUMN dr_id INT NULL COMMENT 'Reference to tbl_doctor.id'`);
        } catch (err) {
            // Column might already exist, ignore error
            if (!err.message.includes('Duplicate column name')) {
                console.log('Note: dr_id column may already exist');
            }
        }

        // Some MySQL configurations don't allow placeholders in LIMIT; inline the sanitized integer
        const sql = `SELECT a.id, 
                            a.name as patient_name, 
                            a.phone, 
                            a.case_ as notes, 
                            a.dr_id,
                            COALESCE(a.dr_name, d.name) as doctor_name, 
                            a.token_number,
                            DATE(a.\`time\`) as date,
                            TIME(a.\`time\`) as time,
                            a.\`time\` as datetime,
                            COALESCE(a.status, 'scheduled') as status
                     FROM tbl_appointment a
                     LEFT JOIN tbl_doctor d ON a.dr_id = d.id
                     ORDER BY CASE WHEN a.\`time\` IS NULL OR a.\`time\` = '' THEN 1 ELSE 0 END, a.\`time\` DESC 
                     LIMIT ${lim}`;
        let appointments;
        try {
            appointments = await query(sql);
        } catch (innerErr) {
            // Fallback: order by id if ordering by time fails due to bad legacy values
            const fallbackSql = `SELECT a.id, 
                                        a.name as patient_name, 
                                        a.phone, 
                                        a.case_ as notes, 
                                        a.dr_id,
                                        COALESCE(a.dr_name, d.name) as doctor_name, 
                                        a.token_number,
                                        DATE(a.\`time\`) as date,
                                        TIME(a.\`time\`) as time,
                                        a.\`time\` as datetime,
                                        COALESCE(a.status, 'scheduled') as status
                                 FROM tbl_appointment a
                                 LEFT JOIN tbl_doctor d ON a.dr_id = d.id
                                 ORDER BY a.id DESC 
                                 LIMIT ${lim}`;
            appointments = await query(fallbackSql);
        }
        res.json({ success: true, data: appointments });
    } catch (error) {
        next(error);
    }
});

// Create appointment
router.post('/', async (req, res, next) => {
    try {
        const { patient_name, name, phone, date, time, dr_id, notes, case_, dr_name, token_number } = req.body || {};

        // Use patient_name or name (support both field names)
        const patientName = patient_name || name;

        // Combine date and time into DATETIME format
        let appointmentDateTime = null;
        if (date && time) {
            // Check if time already has seconds (HH:MM:SS format)
            const timeHasSeconds = time.split(':').length === 3;
            appointmentDateTime = timeHasSeconds ? `${date} ${time}` : `${date} ${time}:00`;
        } else if (time && time.includes('-')) {
            // If time is already a full datetime
            appointmentDateTime = time;
        }

        // Get doctor name if dr_id is provided
        let doctorName = dr_name;
        if (dr_id && !dr_name) {
            try {
                const doctors = await query('SELECT name FROM tbl_doctor WHERE id = ?', [dr_id]);
                if (doctors && doctors.length > 0) {
                    doctorName = doctors[0].name;
                }
            } catch (err) {
                console.log('Could not fetch doctor name:', err.message);
            }
        }

        // Use notes or case_ (support both field names)
        const caseNotes = notes || case_;

        // Coerce undefined -> null (mysql2 disallows undefined in bindings)
        const params = [
            patientName ?? null,
            phone ?? null,
            caseNotes ?? null,
            doctorName ?? null,
            token_number ?? null,
            appointmentDateTime ?? null
        ];

        const sql = 'INSERT INTO tbl_appointment (name, phone, case_, dr_id, dr_name, token_number, time) VALUES (?, ?, ?, ?, ?, ?, ?)';
        const paramsWithDrId = [
            patientName ?? null,
            phone ?? null,
            caseNotes ?? null,
            dr_id ?? null,
            doctorName ?? null,
            token_number ?? null,
            appointmentDateTime ?? null
        ];
        const result = await query(sql, paramsWithDrId);
        res.status(201).json({ success: true, data: { id: result.insertId } });
    } catch (error) {
        next(error);
    }
});

// Get single appointment by ID
router.get('/:id', async (req, res, next) => {
    try {
        const { id } = req.params;
        const sql = `SELECT a.id, 
                            a.name as patient_name, 
                            a.phone, 
                            a.case_ as notes, 
                            a.dr_id,
                            COALESCE(a.dr_name, d.name) as doctor_name, 
                            a.token_number,
                            DATE(a.\`time\`) as date,
                            TIME(a.\`time\`) as time,
                            a.\`time\` as datetime,
                            COALESCE(a.status, 'scheduled') as status
                     FROM tbl_appointment a
                     LEFT JOIN tbl_doctor d ON a.dr_id = d.id
                     WHERE a.id = ?`;
        const appointments = await query(sql, [id]);
        
        if (appointments.length === 0) {
            return res.status(404).json({ success: false, message: 'Appointment not found' });
        }
        
        res.json({ success: true, data: appointments[0] });
    } catch (error) {
        next(error);
    }
});

// Update appointment
router.put('/:id', async (req, res, next) => {
    try {
        const { id } = req.params;
        const { patient_name, name, phone, date, time, dr_id, notes, case_, dr_name, token_number } = req.body || {};

        // Use patient_name or name (support both field names)
        const patientName = patient_name || name;

        // Combine date and time into DATETIME format
        let appointmentDateTime = null;
        if (date && time) {
            // Check if time already has seconds (HH:MM:SS format)
            const timeHasSeconds = time.split(':').length === 3;
            appointmentDateTime = timeHasSeconds ? `${date} ${time}` : `${date} ${time}:00`;
        } else if (time && time.includes('-')) {
            appointmentDateTime = time;
        }

        // Get doctor name if dr_id is provided
        let doctorName = dr_name;
        if (dr_id && !dr_name) {
            try {
                const doctors = await query('SELECT name FROM tbl_doctor WHERE id = ?', [dr_id]);
                if (doctors && doctors.length > 0) {
                    doctorName = doctors[0].name;
                }
            } catch (err) {
                console.log('Could not fetch doctor name:', err.message);
            }
        }

        // Use notes or case_ (support both field names)
        const caseNotes = notes || case_;

        const paramsWithDrId = [
            patientName ?? null,
            phone ?? null,
            caseNotes ?? null,
            dr_id ?? null,
            doctorName ?? null,
            token_number ?? null,
            appointmentDateTime ?? null,
            id
        ];
        const sql = 'UPDATE tbl_appointment SET name = ?, phone = ?, case_ = ?, dr_id = ?, dr_name = ?, token_number = ?, time = ? WHERE id = ?';
        const result = await query(sql, paramsWithDrId);
        
        if (result.affectedRows === 0) {
            return res.status(404).json({ success: false, message: 'Appointment not found' });
        }
        
        res.json({ success: true, message: 'Appointment updated successfully' });
    } catch (error) {
        next(error);
    }
});

// Update appointment status
router.patch('/:id/status', async (req, res, next) => {
    try {
        const { id } = req.params;
        const { status } = req.body;

        // Validate status
        const validStatuses = ['scheduled', 'arrived', 'in-progress', 'completed', 'cancelled', 'no-show'];
        if (!status || !validStatuses.includes(status)) {
            return res.status(400).json({ 
                success: false, 
                message: 'Invalid status. Must be one of: ' + validStatuses.join(', ') 
            });
        }

        const sql = 'UPDATE tbl_appointment SET status = ? WHERE id = ?';
        const result = await query(sql, [status, id]);
        
        if (result.affectedRows === 0) {
            return res.status(404).json({ success: false, message: 'Appointment not found' });
        }
        
        res.json({ 
            success: true, 
            message: `Appointment status updated to ${status}`,
            data: { id, status }
        });
    } catch (error) {
        next(error);
    }
});

// Delete appointment
router.delete('/:id', async (req, res, next) => {
    try {
        const { id } = req.params;
        const sql = 'DELETE FROM tbl_appointment WHERE id = ?';
        const result = await query(sql, [id]);
        
        if (result.affectedRows === 0) {
            return res.status(404).json({ success: false, message: 'Appointment not found' });
        }
        
        res.json({ success: true, message: 'Appointment deleted successfully' });
    } catch (error) {
        next(error);
    }
});

module.exports = router;

