/**
 * Generic CRUD Routes for All Tables
 * Provides REST API for all 46 database tables
 */

const express = require('express');
const router = express.Router();
const { protect } = require('../middleware/auth');
const { requireCrudPermission } = require('../middleware/permissions');
const controllers = require('../controllers/all-crud.controller');
const stockExpenseController = require('../controllers/stock-expense.controller');

// Protect all routes
router.use(protect);

// Define routes for each controller
const resources = [
    'alert', 'appointment', 'case', 'catalog', 'catalog2', 'catExpenses',
    'customer', 'customerDebt', 'expenses', 'form', 'images',
    'inpatient', 'inpatientSale', 'inpatientSaleDebit',
    'labour', 'labourBalance', 'labourLevel', 'loss',
    'patientSale', 'paymentMethod', 'purchase', 'purchaseReturn',
    'purchaseVoucher', 'role', 'sale', 'saleArchive', 'saleDraft',
    'saleDraftNo', 'salePackage', 'salePackageNo', 'saleReturn',
    'saleReturnItem', 'service', 'serviceCatalog', 'shopInfo',
    'stock', 'stockExpense', 'supplier', 'supplierDebt', 'treatment',
    'unit', 'user', 'userLog', 'voucher', 'voucherArchive', 'voucherReturn'
];

// Map CRUD resources to Permission Management resources
const permissionResourceMap = {
    purchase: 'purchases',
    purchaseReturn: 'purchases',
    purchaseVoucher: 'purchases',
    sale: 'sales',
    saleArchive: 'sales',
    saleDraft: 'sales',
    saleDraftNo: 'sales',
    salePackage: 'sales',
    salePackageNo: 'sales',
    saleReturn: 'sales',
    saleReturnItem: 'sales',
    patientSale: 'sales',
    inpatientSale: 'sales',
    inpatientSaleDebit: 'sales',
    stock: 'stock',
    stockExpense: 'stockExpense',
    supplier: 'suppliers',
    expenses: 'expenses',
    appointment: 'appointments',
    inpatient: 'inpatients',
    patient: 'patients',
    doctor: 'doctors',
    user: 'users'
};

const resolvePermissionResource = (resource) => permissionResourceMap[resource] || resource;

// Create routes for each resource
resources.forEach(resource => {
    // Use custom controller for stockExpense
    if (resource === 'stockExpense') {
        const permResource = resolvePermissionResource(resource);
        router.get(`/${resource}`, requireCrudPermission(permResource), stockExpenseController.getAll);
        router.get(`/${resource}/:id`, requireCrudPermission(permResource), stockExpenseController.getOne);
        router.post(`/${resource}`, requireCrudPermission(permResource), stockExpenseController.create);
        router.put(`/${resource}/:id`, requireCrudPermission(permResource), stockExpenseController.update);
        router.delete(`/${resource}/:id`, requireCrudPermission(permResource), stockExpenseController.delete);
    } else {
        const controller = controllers[resource];
        if (controller) {
            const permResource = resolvePermissionResource(resource);
            router.get(`/${resource}`, requireCrudPermission(permResource), controller.getAll);
            router.get(`/${resource}/:id`, requireCrudPermission(permResource), controller.getOne);
            router.post(`/${resource}`, requireCrudPermission(permResource), controller.create);
            router.put(`/${resource}/:id`, requireCrudPermission(permResource), controller.update);
            router.delete(`/${resource}/:id`, requireCrudPermission(permResource), controller.delete);
        }
    }
});

module.exports = router;

