const express = require('express');
const router = express.Router();
const { protect } = require('../middleware/auth');
const { requireCrudPermission } = require('../middleware/permissions');
const { query } = require('../config/database');

router.use(protect);
router.use(requireCrudPermission('dashboard'));

// Get dashboard statistics
router.get('/stats', async (req, res, next) => {
    try {
        const { start_date, end_date } = req.query;
        
        // Determine date range - use provided dates or default to today
        let dateCondition = 'Date = CURDATE()';
        let patientDateCondition = 'date = CURDATE()';
        const salesParams = [];
        const patientParams = [];
        
        if (start_date && end_date) {
            // Use provided date range
            dateCondition = 'Date BETWEEN ? AND ?';
            patientDateCondition = 'date BETWEEN ? AND ?';
            salesParams.push(start_date, end_date);
            patientParams.push(start_date, end_date);
        }
        
        // Sales query with date range
        const salesSql = `SELECT 
                            COALESCE(SUM(Total), 0) as total_sales, 
                            COALESCE(SUM(Profit), 0) as total_profit, 
                            COUNT(DISTINCT VNo) as total_vouchers
                          FROM tbl_sale WHERE ${dateCondition}`;
        const salesResult = await query(salesSql, salesParams);

        // Patients query with date range
        const patientsSql = `SELECT COUNT(*) as total_patients FROM tbl_patient WHERE ${patientDateCondition}`;
        const patientsResult = await query(patientsSql, patientParams);

        // Low stock alerts (not date-dependent)
        const stockSql = 'SELECT COUNT(*) as low_stock FROM tbl_stock WHERE qty <= qty_alert AND qty > 0';
        const stockResult = await query(stockSql);

        // Expiring soon (not date-dependent)
        const expirySql = 'SELECT COUNT(*) as expiring_soon FROM tbl_stock WHERE expire_date BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 30 DAY)';
        const expiryResult = await query(expirySql);

        // Handle NULL values from SUM() - convert to 0
        const totalSales = salesResult[0]?.total_sales ?? 0;
        const totalProfit = salesResult[0]?.total_profit ?? 0;
        const totalVouchers = salesResult[0]?.total_vouchers ?? 0;
        const totalPatients = patientsResult[0]?.total_patients ?? 0;
        
        res.json({
            success: true,
            data: {
                today_sales: totalSales,
                total_sales: totalSales,
                today_profit: totalProfit,
                total_profit: totalProfit,
                today_vouchers: totalVouchers,
                total_vouchers: totalVouchers,
                today_patients: totalPatients,
                total_patients: totalPatients,
                low_stock_count: stockResult[0]?.low_stock || 0,
                expiring_soon_count: expiryResult[0]?.expiring_soon || 0
            }
        });
    } catch (error) {
        next(error);
    }
});

// Get recent activities
router.get('/recent', async (req, res, next) => {
    try {
        const { start_date, end_date, limit } = req.query;
        
        let sql;
        let params = [];
        
        // Build date condition for both parts of UNION
        let voucherDateCondition, inpatientDateCondition;
        if (start_date && end_date) {
            voucherDateCondition = 'DATE(v.Date) BETWEEN ? AND ?';
            inpatientDateCondition = 'DATE(s.Date) BETWEEN ? AND ?';
            params.push(start_date, end_date, start_date, end_date);
        } else {
            voucherDateCondition = 'v.Date >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)';
            inpatientDateCondition = 's.Date >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)';
        }
        
        // Query to get recent sales from vouchers and inpatient sales
        sql = `
            SELECT 
                v.id AS VNo,
                v.Date,
                COALESCE(v.actual_cost, 0) AS total,
                COALESCE(v.user_name, 'System') AS Cashier
            FROM tbl_voucher v
            WHERE ${voucherDateCondition}
              AND v.actual_cost IS NOT NULL
              AND v.actual_cost > 0
            
            UNION ALL
            
            SELECT 
                s.VNo,
                s.Date,
                COALESCE(SUM(s.Total), 0) AS total,
                COALESCE(MAX(s.Cashier), 'System') AS Cashier
            FROM tbl_sale s
            WHERE s.SaleType = 'Inpatient'
              AND s.Barcode NOT LIKE 'BED-%'
              AND s.Barcode NOT LIKE 'DISC-%'
              AND s.Barcode NOT LIKE 'TAX-%'
              AND ${inpatientDateCondition}
              AND s.Total IS NOT NULL
              AND s.Total > 0
            GROUP BY s.VNo, DATE(s.Date)
            HAVING total > 0
            
            ORDER BY Date DESC, VNo DESC
            LIMIT ?
        `;
        
        const limitValue = parseInt(limit) || 10;
        params.push(limitValue);
        
        const recent = await query(sql, params);
        res.json({ success: true, data: recent || [] });
    } catch (error) {
        console.error('Error fetching recent sales:', error);
        next(error);
    }
});

// Get daily sales chart data
router.get('/daily-sales', async (req, res, next) => {
    try {
        const days = parseInt(req.query.days) || 7;
        
        const sql = `
            SELECT 
                DATE(Date) as date,
                SUM(Total) as total_sales,
                COUNT(DISTINCT VNo) as transaction_count
            FROM tbl_sale 
            WHERE Date >= DATE_SUB(CURDATE(), INTERVAL ? DAY)
            GROUP BY DATE(Date)
            ORDER BY DATE(Date) ASC
        `;
        
        const results = await query(sql, [days]);
        
        // Fill in missing dates with zero sales
        const filledData = [];
        const today = new Date();
        for (let i = days - 1; i >= 0; i--) {
            const checkDate = new Date(today);
            checkDate.setDate(today.getDate() - i);
            const dateStr = checkDate.toISOString().split('T')[0];
            
            const existingData = results.find(r => {
                const rDate = new Date(r.date);
                return rDate.toISOString().split('T')[0] === dateStr;
            });
            
            filledData.push({
                date: dateStr,
                total_sales: existingData ? parseFloat(existingData.total_sales) : 0,
                transaction_count: existingData ? parseInt(existingData.transaction_count) : 0
            });
        }
        
        res.json({ 
            success: true, 
            data: filledData 
        });
    } catch (error) {
        next(error);
    }
});

// Get daily patients chart data
router.get('/daily-patients', async (req, res, next) => {
    try {
        const days = parseInt(req.query.days) || 7;
        
        const sql = `
            SELECT 
                DATE(date) as date,
                COUNT(*) as patient_count
            FROM tbl_patient 
            WHERE date >= DATE_SUB(CURDATE(), INTERVAL ? DAY)
            GROUP BY DATE(date)
            ORDER BY DATE(date) ASC
        `;
        
        const results = await query(sql, [days]);
        
        // Fill in missing dates with zero patients
        const filledData = [];
        const today = new Date();
        for (let i = days - 1; i >= 0; i--) {
            const checkDate = new Date(today);
            checkDate.setDate(today.getDate() - i);
            const dateStr = checkDate.toISOString().split('T')[0];
            
            const existingData = results.find(r => {
                const rDate = new Date(r.date);
                return rDate.toISOString().split('T')[0] === dateStr;
            });
            
            filledData.push({
                date: dateStr,
                patient_count: existingData ? parseInt(existingData.patient_count) : 0
            });
        }
        
        res.json({ 
            success: true, 
            data: filledData 
        });
    } catch (error) {
        next(error);
    }
});

// Get daily profit chart data
router.get('/daily-profit', async (req, res, next) => {
    try {
        const days = parseInt(req.query.days) || 7;
        
        const sql = `
            SELECT 
                DATE(Date) as date,
                SUM(Profit) as profit
            FROM tbl_sale 
            WHERE Date >= DATE_SUB(CURDATE(), INTERVAL ? DAY)
            GROUP BY DATE(Date)
            ORDER BY DATE(Date) ASC
        `;
        
        const results = await query(sql, [days]);
        
        // Fill in missing dates with zero profit
        const filledData = [];
        const today = new Date();
        for (let i = days - 1; i >= 0; i--) {
            const checkDate = new Date(today);
            checkDate.setDate(today.getDate() - i);
            const dateStr = checkDate.toISOString().split('T')[0];
            
            const existingData = results.find(r => {
                const rDate = new Date(r.date);
                return rDate.toISOString().split('T')[0] === dateStr;
            });
            
            filledData.push({
                date: dateStr,
                profit: existingData ? parseFloat(existingData.profit) : 0
            });
        }
        
        res.json({ 
            success: true, 
            data: filledData 
        });
    } catch (error) {
        next(error);
    }
});

// Get daily purchase chart data
router.get('/daily-purchase', async (req, res, next) => {
    try {
        const days = parseInt(req.query.days) || 7;
        
        const sql = `
            SELECT 
                DATE(p.date) as date,
                SUM(p.total_cost_price) as total_amount
            FROM tbl_purchase p
            WHERE p.date >= DATE_SUB(CURDATE(), INTERVAL ? DAY)
            GROUP BY DATE(p.date)
            ORDER BY DATE(p.date) ASC
        `;
        
        const results = await query(sql, [days]);
        
        // Fill in missing dates with zero purchase
        const filledData = [];
        const today = new Date();
        for (let i = days - 1; i >= 0; i--) {
            const checkDate = new Date(today);
            checkDate.setDate(today.getDate() - i);
            const dateStr = checkDate.toISOString().split('T')[0];
            
            const existingData = results.find(r => {
                const rDate = new Date(r.date);
                return rDate.toISOString().split('T')[0] === dateStr;
            });
            
            filledData.push({
                date: dateStr,
                total_amount: existingData ? parseFloat(existingData.total_amount) : 0
            });
        }
        
        res.json({ 
            success: true, 
            data: filledData 
        });
    } catch (error) {
        next(error);
    }
});

// Get daily inpatient chart data
router.get('/daily-inpatient', async (req, res, next) => {
    try {
        const days = parseInt(req.query.days) || 7;
        
        const sql = `
            SELECT 
                DATE(start_date) as date,
                COUNT(*) as count
            FROM tbl_inpatient
            WHERE start_date >= DATE_SUB(CURDATE(), INTERVAL ? DAY)
            GROUP BY DATE(start_date)
            ORDER BY DATE(start_date) ASC
        `;
        
        const results = await query(sql, [days]);
        
        // Fill in missing dates with zero count
        const filledData = [];
        const today = new Date();
        for (let i = days - 1; i >= 0; i--) {
            const checkDate = new Date(today);
            checkDate.setDate(today.getDate() - i);
            const dateStr = checkDate.toISOString().split('T')[0];
            
            const existingData = results.find(r => {
                const rDate = new Date(r.date);
                return rDate.toISOString().split('T')[0] === dateStr;
            });
            
            filledData.push({
                date: dateStr,
                count: existingData ? parseInt(existingData.count) : 0
            });
        }
        
        res.json({ 
            success: true, 
            data: filledData 
        });
    } catch (error) {
        next(error);
    }
});

// Get daily laboratory test chart data
router.get('/daily-lab-test', async (req, res, next) => {
    try {
        const days = parseInt(req.query.days) || 7;
        
        const sql = `
            SELECT 
                DATE(created_at) as date,
                COUNT(*) as count
            FROM tbl_lab_report
            WHERE created_at >= DATE_SUB(CURDATE(), INTERVAL ? DAY)
            GROUP BY DATE(created_at)
            ORDER BY DATE(created_at) ASC
        `;
        
        const results = await query(sql, [days]);
        
        // Fill in missing dates with zero count
        const filledData = [];
        const today = new Date();
        for (let i = days - 1; i >= 0; i--) {
            const checkDate = new Date(today);
            checkDate.setDate(today.getDate() - i);
            const dateStr = checkDate.toISOString().split('T')[0];
            
            const existingData = results.find(r => {
                const rDate = new Date(r.date);
                return rDate.toISOString().split('T')[0] === dateStr;
            });
            
            filledData.push({
                date: dateStr,
                count: existingData ? parseInt(existingData.count) : 0
            });
        }
        
        res.json({ 
            success: true, 
            data: filledData 
        });
    } catch (error) {
        next(error);
    }
});

// Get daily expense chart data (combines tbl_expenses and tbl_stock_expense)
router.get('/daily-expense', async (req, res, next) => {
    try {
        const days = parseInt(req.query.days) || 7;
        
        // Combine expenses from both tbl_expenses and tbl_stock_expense
        const sql = `
            SELECT 
                expense_date as date,
                SUM(amount) as total
            FROM (
                SELECT DATE(date) as expense_date, COALESCE(amount, 0) as amount 
                FROM tbl_expenses
                WHERE date >= DATE_SUB(CURDATE(), INTERVAL ? DAY)
                UNION ALL
                SELECT DATE(date) as expense_date, COALESCE(total_cost, 0) as amount 
                FROM tbl_stock_expense
                WHERE date >= DATE_SUB(CURDATE(), INTERVAL ? DAY)
            ) combined_expenses
            GROUP BY expense_date
            ORDER BY expense_date ASC
        `;
        
        const results = await query(sql, [days, days]);
        
        // Fill in missing dates with zero expense
        const filledData = [];
        const today = new Date();
        for (let i = days - 1; i >= 0; i--) {
            const checkDate = new Date(today);
            checkDate.setDate(today.getDate() - i);
            const dateStr = checkDate.toISOString().split('T')[0];
            
            const existingData = results.find(r => {
                const rDate = new Date(r.date);
                return rDate.toISOString().split('T')[0] === dateStr;
            });
            
            filledData.push({
                date: dateStr,
                total: existingData ? parseFloat(existingData.total) : 0
            });
        }
        
        res.json({ 
            success: true, 
            data: filledData 
        });
    } catch (error) {
        next(error);
    }
});

module.exports = router;

