const express = require('express');
const router = express.Router();
const { protect } = require('../middleware/auth');
const { query } = require('../config/database');

// Ensure table exists (safety if migration not yet run)
async function ensureTable() {
    const sql = `
    CREATE TABLE IF NOT EXISTS tbl_delete_history (
        id BIGINT NOT NULL AUTO_INCREMENT,
        table_name VARCHAR(100) NOT NULL,
        record_id VARCHAR(100) NULL,
        deleted_by INT NULL,
        username VARCHAR(100) NULL,
        old_data LONGTEXT NULL COMMENT 'JSON data stored as text',
        reason VARCHAR(255) NULL,
        ip_address VARCHAR(45) NULL,
        user_agent VARCHAR(255) NULL,
        deleted_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        INDEX idx_table (table_name),
        INDEX idx_deleted_by (deleted_by),
        INDEX idx_deleted_at (deleted_at)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
    `;
    await query(sql);
}

router.use(protect);

// GET /api/delete-history
// Query params: start (YYYY-MM-DD), end (YYYY-MM-DD), table (string), user (id or username), limit (default 200)
router.get('/', async (req, res, next) => {
    try {
        const { start, end, table, user, search, limit } = req.query;
        const params = [];
        let sql = `
            SELECT id, table_name, record_id, deleted_by, username, reason, ip_address, user_agent, old_data, deleted_at
            FROM tbl_delete_history
            WHERE 1=1
        `;

        if (start) {
            sql += ' AND deleted_at >= ?';
            params.push(start + ' 00:00:00');
        }
        if (end) {
            sql += ' AND deleted_at <= ?';
            params.push(end + ' 23:59:59');
        }
        if (table) {
            sql += ' AND table_name = ?';
            params.push(table);
        }
        if (user) {
            sql += ' AND (deleted_by = ? OR username LIKE ?)';
            params.push(user, `%${user}%`);
        }
        if (search) {
            sql += ' AND (record_id LIKE ? OR reason LIKE ?)';
            params.push(`%${search}%`, `%${search}%`);
        }

        sql += ' ORDER BY deleted_at DESC';
        const maxLimit = Math.min(parseInt(limit) || 200, 500);
        sql += ` LIMIT ${maxLimit}`;

        await ensureTable();
        const rows = await query(sql, params);

        res.json({
            success: true,
            data: rows,
        });
    } catch (error) {
        console.error('Delete history fetch error:', error);
        next(error);
    }
});

module.exports = router;

