/**
 * Health Check Routes
 * Monitor system health and database pool status
 */

const express = require('express');
const router = express.Router();
const { getPoolStatus } = require('../config/database');

/**
 * @desc    Get system health status
 * @route   GET /api/health
 * @access  Public
 */
router.get('/', (req, res) => {
    const poolStatus = getPoolStatus();
    
    const health = {
        status: 'healthy',
        timestamp: new Date().toISOString(),
        uptime: process.uptime(),
        memory: {
            used: Math.round(process.memoryUsage().heapUsed / 1024 / 1024),
            total: Math.round(process.memoryUsage().heapTotal / 1024 / 1024),
            rss: Math.round(process.memoryUsage().rss / 1024 / 1024)
        },
        database: {
            ...poolStatus,
            status: poolStatus.activeConnections < 45 ? 'healthy' : 'warning'
        }
    };

    // Set status based on conditions
    if (poolStatus.activeConnections >= 45) {
        health.status = 'degraded';
        health.warning = 'High database connection usage';
    }
    
    if (poolStatus.queuedRequests > 10) {
        health.status = 'unhealthy';
        health.error = 'Many queued database requests';
    }

    const statusCode = health.status === 'healthy' ? 200 : 
                       health.status === 'degraded' ? 429 : 503;

    res.status(statusCode).json(health);
});

/**
 * @desc    Get database pool status
 * @route   GET /api/health/database
 * @access  Public
 */
router.get('/database', (req, res) => {
    const poolStatus = getPoolStatus();
    
    res.json({
        success: true,
        data: {
            ...poolStatus,
            connectionLimit: 50,
            utilizationPercent: Math.round((poolStatus.activeConnections / 50) * 100)
        }
    });
});

module.exports = router;

