/**
 * Hospital Operations Routes
 * Bed Management, OR Scheduling, Emergency Department
 */

const express = require('express');
const router = express.Router();
const { protect } = require('../middleware/auth');
const {
    getBeds,
    getBedAvailability,
    assignBed,
    dischargeBed,
    createBed,
    updateBed,
    deleteBed,
    getOperatingRooms,
    scheduleSurgery,
    getSurgerySchedules,
    updateSurgeryStatus,
    createEmergencyAdmission,
    getEmergencyAdmissions,
    updateEmergencyAdmission,
    getDepartments,
    createDepartment,
    updateDepartment,
    deleteDepartment,
    getHospitalStats
} = require('../controllers/hospital.controller');

// Log route registration for debugging
console.log('Hospital routes module loaded');

// Health check route (no auth required for testing)
router.get('/health', (req, res) => {
    res.json({ 
        success: true, 
        message: 'Hospital routes are working',
        timestamp: new Date().toISOString()
    });
});

router.use(protect);

// Departments
router.get('/departments', getDepartments);
router.post('/departments', createDepartment);
router.put('/departments/:id', updateDepartment);
router.delete('/departments/:id', deleteDepartment);

// Bed Management
router.get('/beds', getBeds);
router.get('/beds/availability', getBedAvailability);
router.post('/beds/assign', assignBed);
router.patch('/beds/:bedId/discharge', dischargeBed);
router.post('/beds', createBed);
router.put('/beds/:bedId', updateBed);
router.delete('/beds/:bedId', deleteBed);

// Operating Rooms
router.get('/operating-rooms', getOperatingRooms);
router.post('/surgeries', scheduleSurgery);
router.get('/surgeries', getSurgerySchedules);
router.patch('/surgeries/:id/status', updateSurgeryStatus);

// Emergency Department
router.post('/emergency', createEmergencyAdmission);
router.get('/emergency', getEmergencyAdmissions);
router.patch('/emergency/:id', updateEmergencyAdmission);

// Statistics
router.get('/stats', getHospitalStats);

module.exports = router;

