/**
 * Lab Order Routes
 * Handles lab orders created from sales
 */

const express = require('express');
const router = express.Router();
const { protect } = require('../middleware/auth');
const { query } = require('../config/database');

router.use(protect);

/**
 * @desc    Get all lab orders (pending orders only by default)
 * @route   GET /api/lab-orders
 * @access  Private
 */
router.get('/', async (req, res, next) => {
    try {
        const { status, patient_id, doctor_id } = req.query;
        
        let sql = `SELECT lo.*, 
                          p.name as patient_full_name, p.age as patient_age, p.address as patient_address,
                          d.name as doctor_full_name, d.specialist as doctor_specialist
                   FROM tbl_lab_order lo
                   LEFT JOIN tbl_patient p ON lo.patient_id = p.patient_id
                   LEFT JOIN tbl_doctor d ON lo.doctor_id = d.id
                   WHERE 1=1`;
        
        const params = [];
        
        // Filter by status (default to non-completed orders)
        if (status) {
            sql += ' AND lo.status = ?';
            params.push(status);
        } else {
            sql += ' AND lo.status != "Completed"';
        }
        
        if (patient_id) {
            sql += ' AND lo.patient_id = ?';
            params.push(patient_id);
        }
        
        if (doctor_id) {
            sql += ' AND lo.doctor_id = ?';
            params.push(doctor_id);
        }
        
        sql += ' ORDER BY lo.order_date DESC LIMIT 200';
        
        const orders = await query(sql, params);
        
        res.json({
            success: true,
            count: orders.length,
            data: orders
        });
    } catch (error) {
        console.error('Error fetching lab orders:', error);
        res.status(500).json({
            success: false,
            message: 'Error fetching lab orders: ' + error.message
        });
    }
});

/**
 * @desc    Get single lab order
 * @route   GET /api/lab-orders/:id
 * @access  Private
 */
router.get('/:id', async (req, res, next) => {
    try {
        const sql = `SELECT lo.*, 
                           p.name as patient_full_name, p.age as patient_age, p.address as patient_address,
                           p.phone as patient_phone,
                           d.name as doctor_full_name, d.specialist as doctor_specialist,
                           d.phone as doctor_phone
                    FROM tbl_lab_order lo
                    LEFT JOIN tbl_patient p ON lo.patient_id = p.patient_id OR (lo.patient_id = 'Walk-in' AND p.patient_id IS NULL)
                    LEFT JOIN tbl_doctor d ON lo.doctor_id = d.id
                    WHERE lo.id = ?`;
        
        const orders = await query(sql, [req.params.id]);
        
        if (!orders || orders.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Lab order not found'
            });
        }
        
        // Use patient_name from order if patient_full_name is not available
        const order = orders[0];
        if (!order.patient_full_name && order.patient_name) {
            order.patient_full_name = order.patient_name;
        }
        
        res.json({
            success: true,
            data: order
        });
    } catch (error) {
        console.error('Error fetching lab order:', error);
        console.error('Error stack:', error.stack);
        res.status(500).json({
            success: false,
            message: 'Error fetching lab order: ' + error.message
        });
    }
});

/**
 * @desc    Update lab order status
 * @route   PUT /api/lab-orders/:id/status
 * @access  Private
 */
router.put('/:id/status', async (req, res, next) => {
    try {
        const { status, collection_date, collected_by, notes } = req.body;
        
        if (!status) {
            return res.status(400).json({
                success: false,
                message: 'Status is required'
            });
        }
        
        const validStatuses = ['Ordered', 'Specimen_Collected', 'In_Progress', 'Completed', 'Cancelled'];
        if (!validStatuses.includes(status)) {
            return res.status(400).json({
                success: false,
                message: 'Invalid status'
            });
        }
        
        let sql = 'UPDATE tbl_lab_order SET status = ?';
        const params = [status];
        
        if (collection_date) {
            sql += ', collection_date = ?';
            params.push(collection_date);
        }
        
        if (collected_by) {
            sql += ', collected_by = ?';
            params.push(collected_by);
        }
        
        if (notes !== undefined) {
            sql += ', notes = ?';
            params.push(notes);
        }
        
        sql += ' WHERE id = ?';
        params.push(req.params.id);
        
        const result = await query(sql, params);
        
        if (result.affectedRows === 0) {
            return res.status(404).json({
                success: false,
                message: 'Lab order not found'
            });
        }
        
        res.json({
            success: true,
            message: 'Lab order status updated successfully'
        });
    } catch (error) {
        console.error('Error updating lab order status:', error);
        res.status(500).json({
            success: false,
            message: 'Error updating lab order status: ' + error.message
        });
    }
});

/**
 * @desc    Create lab report from order
 * @route   POST /api/lab-orders/:id/create-report
 * @access  Private
 */
router.post('/:id/create-report', async (req, res, next) => {
    try {
        const { report_no, report_type, collected_at, reported_at, notes, items } = req.body;
        
        // Get order details
        const [order] = await query('SELECT * FROM tbl_lab_order WHERE id = ?', [req.params.id]);
        
        if (order.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Lab order not found'
            });
        }
        
        const orderData = order[0];
        
        // Create lab report (removed order_id as it doesn't exist in table)
        const reportResult = await query(
            `INSERT INTO tbl_lab_report 
             (report_no, patient_id, doctor_id, report_type, collected_at, reported_at, notes, referring_doctor_id, referring_doctor_name)
             VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)`,
            [
                report_no || null,
                orderData.patient_id,
                orderData.doctor_id,
                report_type || orderData.lab_item_name,
                collected_at || new Date(),
                reported_at || new Date(),
                notes || null,
                orderData.referring_doctor_id || null,
                orderData.referring_doctor_name || null
            ]
        );
        
        const reportId = reportResult.insertId;
        
        // Insert report items
        if (items && Array.isArray(items)) {
            for (const item of items) {
                // Calculate flag based on reference range
                let flag = 'Normal';
                if (item.result_value && item.reference_min !== null && item.reference_max !== null) {
                    const resultNum = parseFloat(item.result_value);
                    if (!isNaN(resultNum)) {
                        if (resultNum < item.reference_min) {
                            flag = 'Low';
                        } else if (resultNum > item.reference_max) {
                            flag = 'High';
                        }
                    }
                }
                
                // Build reference_range string from min/max if provided, otherwise use the reference_range string
                let referenceRangeStr = item.reference_range || '';
                if (!referenceRangeStr && item.reference_min !== null && item.reference_max !== null) {
                    referenceRangeStr = `${item.reference_min} - ${item.reference_max}`;
                }
                
                await query(
                    `INSERT INTO tbl_lab_report_item 
                     (report_id, test_name, result_value, unit, reference_range, flag, remark)
                     VALUES (?, ?, ?, ?, ?, ?, ?)`,
                    [
                        reportId,
                        item.test_name,
                        item.result_value || null,
                        item.unit || null,
                        referenceRangeStr || null,
                        flag,
                        item.remark || null
                    ]
                );
            }
        }
        
        // Update order status to Completed
        await query('UPDATE tbl_lab_order SET status = "Completed" WHERE id = ?', [req.params.id]);
        
        res.json({
            success: true,
            message: 'Lab report created successfully',
            data: {
                report_id: reportId,
                order_id: req.params.id
            }
        });
    } catch (error) {
        console.error('Error creating lab report from order:', error);
        res.status(500).json({
            success: false,
            message: 'Error creating lab report: ' + error.message
        });
    }
});

module.exports = router;

