const express = require('express');
const router = express.Router();
const { query } = require('../config/database');
const { protect } = require('../middleware/auth');
const multer = require('multer');
const path = require('path');
const fs = require('fs');

// Create uploads/signatures directory if it doesn't exist
const signatureUploadDir = path.join(__dirname, '../uploads/signatures');
if (!fs.existsSync(signatureUploadDir)) {
    fs.mkdirSync(signatureUploadDir, { recursive: true });
}

// Storage configuration for signatures
const signatureStorage = multer.diskStorage({
    destination: (req, file, cb) => {
        cb(null, signatureUploadDir);
    },
    filename: (req, file, cb) => {
        const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
        const ext = path.extname(file.originalname);
        cb(null, `signature-${uniqueSuffix}${ext}`);
    }
});

const signatureUpload = multer({
    storage: signatureStorage,
    limits: {
        fileSize: 2 * 1024 * 1024 // 2MB
    },
    fileFilter: (req, file, cb) => {
        const allowedTypes = /jpeg|jpg|png|gif|webp/;
        const extname = allowedTypes.test(path.extname(file.originalname).toLowerCase());
        const mimetype = allowedTypes.test(file.mimetype);
        if (mimetype && extname) {
            return cb(null, true);
        }
        cb(new Error('Only image files are allowed'));
    }
});

// Create lab report with items
router.post('/', protect, async (req, res) => {
    try {
        const { report, items } = req.body; // report: header, items: array
        if (!report || !report.patient_id || !Array.isArray(items)) {
            return res.status(400).json({ success: false, message: 'Invalid request body' });
        }

        // Insert header (removed order_id and doctor_signature as they don't exist in table)
        const result = await query(
            `INSERT INTO tbl_lab_report (report_no, patient_id, doctor_id, report_type, collected_at, reported_at, notes, referring_doctor_id, referring_doctor_name)
             VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)`,
            [
                report.report_no || null,
                report.patient_id,
                report.doctor_id || null,
                report.report_type || null,
                report.collected_at || null,
                report.reported_at || null,
                report.notes || null,
                report.referring_doctor_id || null,
                report.referring_doctor_name || null
            ]
        );
        const reportId = result.insertId;

        // Insert items with auto-flagging
        for (const item of items) {
            // Auto-calculate flag based on reference range
            let flag = item.flag || 'Normal';
            if (item.result_value && item.reference_min !== null && item.reference_max !== null) {
                const resultNum = parseFloat(item.result_value);
                if (!isNaN(resultNum)) {
                    if (resultNum < item.reference_min) {
                        flag = 'Low';
                    } else if (resultNum > item.reference_max) {
                        flag = 'High';
                    } else {
                        flag = 'Normal';
                    }
                }
            }
            
            // Build reference_range string from min/max if provided, otherwise use the reference_range string
            let referenceRangeStr = item.reference_range || '';
            if (!referenceRangeStr && item.reference_min !== null && item.reference_max !== null) {
                referenceRangeStr = `${item.reference_min} - ${item.reference_max}`;
            }
            
            await query(
                `INSERT INTO tbl_lab_report_item (report_id, test_name, result_value, unit, reference_range, flag, remark)
                 VALUES (?, ?, ?, ?, ?, ?, ?)`,
                [
                    reportId,
                    item.test_name,
                    item.result_value || null,
                    item.unit || null,
                    referenceRangeStr || null,
                    flag,
                    item.remark || null
                ]
            );
        }

        res.status(201).json({ success: true, id: reportId });
    } catch (error) {
        res.status(500).json({ success: false, message: error.message });
    }
});

// Get lab report by id including items and linked patient/doctor
router.get('/:id', protect, async (req, res) => {
    try {
        const [header] = await query(
            `SELECT lr.*, p.name AS patient_name, p.patient_id AS patient_code, p.age AS patient_age, p.address AS patient_address,
                    d.name AS doctor_name, d.specialist AS doctor_specialist,
                    lr.referring_doctor_name
             FROM tbl_lab_report lr
             LEFT JOIN tbl_patient p ON lr.patient_id = p.patient_id
             LEFT JOIN tbl_doctor d ON lr.doctor_id = d.id
             WHERE lr.id = ?`,
            [req.params.id]
        );
        if (!header) return res.status(404).json({ success: false, message: 'Not found' });

        const items = await query(
            `SELECT * FROM tbl_lab_report_item WHERE report_id = ? ORDER BY id ASC`,
            [req.params.id]
        );
        
        // Auto-update flags if reference ranges exist and result values are numeric
        // Note: reference_min and reference_max columns don't exist, so we parse from reference_range string
        for (const item of items) {
            if (item.result_value && item.reference_range) {
                const resultNum = parseFloat(item.result_value);
                if (!isNaN(resultNum)) {
                    // Try to parse min/max from reference_range string (e.g., "12 - 16" or "12-16")
                    const rangeMatch = item.reference_range.match(/(\d+\.?\d*)\s*-\s*(\d+\.?\d*)/);
                    if (rangeMatch) {
                        const min = parseFloat(rangeMatch[1]);
                        const max = parseFloat(rangeMatch[2]);
                        if (!isNaN(min) && !isNaN(max)) {
                            let flag = 'Normal';
                            if (resultNum < min) {
                                flag = 'Low';
                            } else if (resultNum > max) {
                                flag = 'High';
                            }
                            
                            // Update flag if different
                            if (item.flag !== flag) {
                                await query(
                                    'UPDATE tbl_lab_report_item SET flag = ? WHERE id = ?',
                                    [flag, item.id]
                                );
                                item.flag = flag;
                            }
                        }
                    }
                }
            }
        }

        res.json({ success: true, data: { report: header, items } });
    } catch (error) {
        res.status(500).json({ success: false, message: error.message });
    }
});

// List lab reports (latest first) with basic info
router.get('/', protect, async (req, res) => {
    try {
        const reports = await query(
            `SELECT lr.id, lr.report_no, lr.report_type, lr.created_at,
                    p.patient_id, p.name AS patient_name,
                    d.name AS doctor_name,
                    lr.referring_doctor_name
             FROM tbl_lab_report lr
             LEFT JOIN tbl_patient p ON lr.patient_id = p.patient_id
             LEFT JOIN tbl_doctor d ON lr.doctor_id = d.id
             ORDER BY lr.id DESC LIMIT 200`
        );
        res.json({ success: true, data: reports });
    } catch (error) {
        res.status(500).json({ success: false, message: error.message });
    }
});

/**
 * @desc    Update lab report with doctor signature
 * @route   PUT /api/lab/:id/signature
 * @access  Private
 */
router.put('/:id/signature', protect, signatureUpload.single('signature'), async (req, res) => {
    try {
        if (!req.file) {
            return res.status(400).json({ success: false, message: 'No signature file uploaded' });
        }
        
        // Save path as /uploads/signatures/filename
        const signaturePath = `/uploads/signatures/${req.file.filename}`;
        
        // Note: doctor_signature column doesn't exist in tbl_lab_report table
        // The signature file is uploaded but not stored in the database
        // You may want to add this column or store signatures elsewhere
        console.log('Signature uploaded to:', signaturePath, 'but not stored in database (column does not exist)');
        
        res.json({
            success: true,
            message: 'Doctor signature uploaded successfully',
            data: { signature: signaturePath }
        });
    } catch (error) {
        console.error('Error uploading signature:', error);
        res.status(500).json({ success: false, message: error.message });
    }
});

module.exports = router;


