/**
 * Laboratory Item Routes
 * Similar to services but for laboratory items
 */

const express = require('express');
const router = express.Router();
const { protect } = require('../middleware/auth');
const { requireCrudPermission } = require('../middleware/permissions');
const { query } = require('../config/database');

router.use(protect);
router.use(requireCrudPermission('laboratory'));

/**
 * @desc    Get all laboratory items
 * @route   GET /api/laboratory-items
 * @access  Private
 */
router.get('/', async (req, res, next) => {
    try {
        const { page = 1, limit = 200, category, search } = req.query;
        const offset = (page - 1) * limit;

        let sql = `SELECT li.*, sc.name as category_name 
                   FROM tbl_laboratory_item li 
                   LEFT JOIN tbl_service_catalog sc ON li.cat_id = sc.id
                   WHERE 1=1`;
        
        const params = [];

        if (category) {
            sql += ' AND li.cat_id = ?';
            params.push(category);
        }

        if (search) {
            sql += ' AND (li.item_name LIKE ? OR li.barcode LIKE ?)';
            params.push(`%${search}%`, `%${search}%`);
        }

        sql += ' ORDER BY li.item_name LIMIT ? OFFSET ?';
        params.push(parseInt(limit), parseInt(offset));

        const items = await query(sql, params);

        // Get total count
        let countSql = `SELECT COUNT(*) as total FROM tbl_laboratory_item li WHERE 1=1`;
        const countParams = [];
        
        if (category) {
            countSql += ' AND li.cat_id = ?';
            countParams.push(category);
        }
        
        if (search) {
            countSql += ' AND (li.item_name LIKE ? OR li.barcode LIKE ?)';
            countParams.push(`%${search}%`, `%${search}%`);
        }

        const countResult = await query(countSql, countParams);

        res.json({
            success: true,
            count: items.length,
            total: countResult[0].total,
            page: parseInt(page),
            pages: Math.ceil(countResult[0].total / limit),
            data: items
        });
    } catch (error) {
        console.error('Error fetching laboratory items:', error);
        res.json({
            success: true,
            data: [],
            total: 0,
            count: 0,
            message: 'No laboratory items found'
        });
    }
});

/**
 * @desc    Get laboratory item categories (uses service categories)
 * @route   GET /api/laboratory-items/categories/list
 * @access  Private
 */
router.get('/categories/list', async (req, res, next) => {
    try {
        const sql = 'SELECT * FROM tbl_service_catalog ORDER BY name';
        const categories = await query(sql);
        
        res.json({
            success: true,
            data: categories
        });
    } catch (error) {
        console.error('Error fetching categories:', error);
        res.json({
            success: true,
            data: []
        });
    }
});

/**
 * @desc    Get single laboratory item
 * @route   GET /api/laboratory-items/:id
 * @access  Private
 */
router.get('/:id', async (req, res, next) => {
    try {
        const sql = `SELECT li.*, sc.name as category_name 
                     FROM tbl_laboratory_item li 
                     LEFT JOIN tbl_service_catalog sc ON li.cat_id = sc.id
                     WHERE li.id = ?`;
        
        const items = await query(sql, [req.params.id]);

        if (items.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Laboratory item not found'
            });
        }

        res.json({
            success: true,
            data: items[0]
        });
    } catch (error) {
        console.error('Error fetching laboratory item:', error);
        res.status(500).json({
            success: false,
            message: 'Error fetching laboratory item details'
        });
    }
});

/**
 * @desc    Create new laboratory item
 * @route   POST /api/laboratory-items
 * @access  Private
 */
router.post('/', async (req, res, next) => {
    try {
        const {
            barcode,
            item_name,
            cat_id,
            buy_price,
            sale_price,
            unit,
            reference_ranges,
            remark
        } = req.body;

        // Debug: Log received data
        console.log('Creating laboratory item - received data:', {
            item_name,
            unit,
            reference_ranges,
            unit_type: typeof unit,
            reference_ranges_type: typeof reference_ranges
        });

        // Validate required fields
        if (!item_name) {
            return res.status(400).json({
                success: false,
                message: 'Item name is required'
            });
        }

        // Check if barcode already exists
        if (barcode) {
            const existingItem = await query('SELECT id FROM tbl_laboratory_item WHERE barcode = ?', [barcode]);
            if (existingItem.length > 0) {
                return res.status(400).json({
                    success: false,
                    message: 'Laboratory item with this barcode already exists'
                });
            }
        }

        const sql = `INSERT INTO tbl_laboratory_item 
                     (barcode, item_name, cat_id, buy_price, sale_price, unit, reference_ranges, remark)
                     VALUES (?, ?, ?, ?, ?, ?, ?, ?)`;

        // Handle unit and reference_ranges - allow empty strings, convert to null only if undefined
        const unitValue = (unit !== undefined && unit !== null && String(unit).trim() !== '') ? String(unit).trim() : null;
        const referenceRangesValue = (reference_ranges !== undefined && reference_ranges !== null && String(reference_ranges).trim() !== '') ? String(reference_ranges).trim() : null;

        console.log('Inserting with values:', {
            unit: unitValue,
            reference_ranges: referenceRangesValue
        });

        const result = await query(sql, [
            barcode || null,
            item_name,
            cat_id || null,
            parseFloat(buy_price) || 0,
            parseFloat(sale_price) || 0,
            unitValue,
            referenceRangesValue,
            remark || null
        ]);

        res.status(201).json({
            success: true,
            message: 'Laboratory item created successfully',
            data: {
                id: result.insertId,
                barcode,
                item_name,
                cat_id,
                buy_price,
                sale_price,
                unit,
                reference_ranges,
                remark
            }
        });
    } catch (error) {
        console.error('Error creating laboratory item:', error);
        res.status(500).json({
            success: false,
            message: 'Error creating laboratory item: ' + error.message
        });
    }
});

/**
 * @desc    Update laboratory item
 * @route   PUT /api/laboratory-items/:id
 * @access  Private
 */
router.put('/:id', async (req, res, next) => {
    try {
        const {
            barcode,
            item_name,
            cat_id,
            buy_price,
            sale_price,
            unit,
            reference_ranges,
            remark
        } = req.body;

        // Validate required fields
        if (!item_name) {
            return res.status(400).json({
                success: false,
                message: 'Item name is required'
            });
        }

        // Get existing item
        const existingItems = await query('SELECT * FROM tbl_laboratory_item WHERE id = ?', [req.params.id]);
        
        if (existingItems.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Laboratory item not found'
            });
        }

        // Check if barcode already exists (excluding current item)
        if (barcode) {
            const existingItem = await query('SELECT id FROM tbl_laboratory_item WHERE barcode = ? AND id != ?', [barcode, req.params.id]);
            if (existingItem.length > 0) {
                return res.status(400).json({
                    success: false,
                    message: 'Laboratory item with this barcode already exists'
                });
            }
        }

        // Debug: Log received data
        console.log('Updating laboratory item - received data:', {
            id: req.params.id,
            item_name,
            unit,
            reference_ranges,
            unit_type: typeof unit,
            reference_ranges_type: typeof reference_ranges
        });

        // Handle unit and reference_ranges - allow empty strings, convert to null only if undefined
        const unitValue = (unit !== undefined && unit !== null && String(unit).trim() !== '') ? String(unit).trim() : null;
        const referenceRangesValue = (reference_ranges !== undefined && reference_ranges !== null && String(reference_ranges).trim() !== '') ? String(reference_ranges).trim() : null;

        console.log('Updating with values:', {
            unit: unitValue,
            reference_ranges: referenceRangesValue
        });

        const sql = `UPDATE tbl_laboratory_item 
                     SET barcode = ?, item_name = ?, cat_id = ?, 
                         buy_price = ?, sale_price = ?, unit = ?, reference_ranges = ?, remark = ?
                     WHERE id = ?`;

        const result = await query(sql, [
            barcode || null,
            item_name,
            cat_id || null,
            parseFloat(buy_price) || 0,
            parseFloat(sale_price) || 0,
            unitValue,
            referenceRangesValue,
            remark || null,
            req.params.id
        ]);

        res.json({
            success: true,
            message: 'Laboratory item updated successfully',
            data: {
                id: req.params.id,
                affectedRows: result.affectedRows
            }
        });
    } catch (error) {
        console.error('Error updating laboratory item:', error);
        res.status(500).json({
            success: false,
            message: 'Error updating laboratory item: ' + error.message
        });
    }
});

/**
 * @desc    Delete laboratory item
 * @route   DELETE /api/laboratory-items/:id
 * @access  Private
 */
router.delete('/:id', async (req, res, next) => {
    try {
        const items = await query('SELECT item_name FROM tbl_laboratory_item WHERE id = ?', [req.params.id]);
        
        if (items.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Laboratory item not found'
            });
        }

        await query('DELETE FROM tbl_laboratory_item WHERE id = ?', [req.params.id]);

        res.json({
            success: true,
            message: 'Laboratory item deleted successfully'
        });
    } catch (error) {
        console.error('Error deleting laboratory item:', error);
        res.status(500).json({
            success: false,
            message: 'Error deleting laboratory item: ' + error.message
        });
    }
});

/**
 * @desc    Generate barcode for laboratory item
 * @route   GET /api/laboratory-items/generate-barcode/new
 * @access  Private
 */
router.get('/generate-barcode/new', async (req, res, next) => {
    try {
        // Fetch recent barcodes and auto-increment the highest numeric value (strip non-digits)
        const rows = await query(
            `SELECT barcode FROM tbl_laboratory_item 
             WHERE barcode IS NOT NULL AND barcode <> '' 
             ORDER BY id DESC 
             LIMIT 5000`
        );

        const toDigits = (val) => String(val || '').replace(/\D/g, '');

        const numericEntries = rows
            .map(r => {
                const digits = toDigits(r.barcode);
                return digits.length ? { digits, len: digits.length, num: parseInt(digits, 10) } : null;
            })
            .filter(e => e && !isNaN(e.num));

        let barcode = '100001'; // default starting point when no barcodes exist

        if (numericEntries.length) {
            const maxEntry = numericEntries.reduce((max, entry) => {
                if (entry.num > max.num) return entry;
                return max;
            }, { num: -1, len: 6, digits: '' });

            const nextNumber = maxEntry.num + 1;
            const targetLength = Math.max(maxEntry.len, String(nextNumber).length);
            barcode = String(nextNumber).padStart(targetLength, '0');
        }

        res.json({ success: true, barcode });
    } catch (error) {
        console.error('Error generating barcode:', error);
        res.status(500).json({
            success: false,
            message: 'Error generating barcode'
        });
    }
});

module.exports = router;

