/**
 * Laboratory Package Routes
 * Manages laboratory packages that contain multiple laboratory items
 */

const express = require('express');
const router = express.Router();
const { protect } = require('../middleware/auth');
const { requireCrudPermission } = require('../middleware/permissions');
const { query, transaction } = require('../config/database');

router.use(protect);
router.use(requireCrudPermission('laboratory'));

/**
 * @desc    Get all laboratory packages
 * @route   GET /api/laboratory-packages
 * @access  Private
 */
router.get('/', async (req, res, next) => {
    try {
        const { page = 1, limit = 200, search, active } = req.query;
        const offset = (page - 1) * limit;

        let sql = `SELECT lp.*, 
                   COUNT(lpi.id) as item_count,
                   SUM(lpi.item_price * lpi.quantity) as calculated_total
                   FROM tbl_laboratory_package lp
                   LEFT JOIN tbl_laboratory_package_item lpi ON lp.id = lpi.package_id
                   WHERE 1=1`;
        
        const params = [];

        if (active !== undefined) {
            sql += ' AND lp.is_active = ?';
            params.push(active === 'true' ? 1 : 0);
        }

        if (search) {
            sql += ' AND (lp.package_name LIKE ? OR lp.package_code LIKE ?)';
            params.push(`%${search}%`, `%${search}%`);
        }

        sql += ' GROUP BY lp.id ORDER BY lp.package_name LIMIT ? OFFSET ?';
        params.push(parseInt(limit), parseInt(offset));

        const packages = await query(sql, params);

        // Get total count
        let countSql = `SELECT COUNT(DISTINCT lp.id) as total 
                       FROM tbl_laboratory_package lp 
                       WHERE 1=1`;
        const countParams = [];
        
        if (active !== undefined) {
            countSql += ' AND lp.is_active = ?';
            countParams.push(active === 'true' ? 1 : 0);
        }
        
        if (search) {
            countSql += ' AND (lp.package_name LIKE ? OR lp.package_code LIKE ?)';
            countParams.push(`%${search}%`, `%${search}%`);
        }

        const countResult = await query(countSql, countParams);

        res.json({
            success: true,
            count: packages.length,
            total: countResult[0].total,
            page: parseInt(page),
            pages: Math.ceil(countResult[0].total / limit),
            data: packages
        });
    } catch (error) {
        console.error('Error fetching laboratory packages:', error);
        res.status(500).json({
            success: false,
            message: 'Error fetching laboratory packages: ' + error.message
        });
    }
});

/**
 * @desc    Get single laboratory package with items
 * @route   GET /api/laboratory-packages/:id
 * @access  Private
 */
router.get('/:id', async (req, res, next) => {
    try {
        // Get package details
        const packageSql = `SELECT * FROM tbl_laboratory_package WHERE id = ?`;
        const packages = await query(packageSql, [req.params.id]);

        if (packages.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Laboratory package not found'
            });
        }

        const packageData = packages[0];

        // Get package items with laboratory item details
        const itemsSql = `SELECT lpi.*, 
                         li.item_name, li.barcode, li.sale_price as current_price,
                         li.unit, li.reference_ranges
                         FROM tbl_laboratory_package_item lpi
                         JOIN tbl_laboratory_item li ON lpi.lab_item_id = li.id
                         WHERE lpi.package_id = ?
                         ORDER BY li.item_name`;
        
        const items = await query(itemsSql, [req.params.id]);

        res.json({
            success: true,
            data: {
                ...packageData,
                items: items
            }
        });
    } catch (error) {
        console.error('Error fetching laboratory package:', error);
        res.status(500).json({
            success: false,
            message: 'Error fetching laboratory package details: ' + error.message
        });
    }
});

/**
 * @desc    Create new laboratory package
 * @route   POST /api/laboratory-packages
 * @access  Private
 */
router.post('/', async (req, res, next) => {
    try {
        const {
            package_name,
            package_code,
            description,
            total_price,
            is_active,
            items // Array of {lab_item_id, quantity, item_price}
        } = req.body;

        // Validate required fields
        if (!package_name) {
            return res.status(400).json({
                success: false,
                message: 'Package name is required'
            });
        }

        if (!items || !Array.isArray(items) || items.length === 0) {
            return res.status(400).json({
                success: false,
                message: 'Package must contain at least one item'
            });
        }

        // Check if package code already exists
        if (package_code) {
            const existing = await query(
                'SELECT id FROM tbl_laboratory_package WHERE package_code = ?',
                [package_code]
            );
            if (existing.length > 0) {
                return res.status(400).json({
                    success: false,
                    message: 'Package code already exists'
                });
            }
        }

        // Use transaction to ensure data consistency
        const result = await transaction(async (conn) => {
            // Insert package
            const packageSql = `INSERT INTO tbl_laboratory_package 
                               (package_name, package_code, description, total_price, is_active)
                               VALUES (?, ?, ?, ?, ?)`;
            
            const packageResult = await conn.query(packageSql, [
                package_name,
                package_code || null,
                description || null,
                parseFloat(total_price) || 0,
                is_active !== undefined ? (is_active ? 1 : 0) : 1
            ]);

            const packageId = packageResult[0].insertId;

            // Calculate total price from items if not provided
            let calculatedTotal = 0;

            // Insert package items
            for (const item of items) {
                if (!item.lab_item_id) {
                    throw new Error('lab_item_id is required for each item');
                }

                // Get current price if not provided
                let itemPrice = parseFloat(item.item_price) || 0;
                if (!itemPrice) {
                    const [labItems] = await conn.query(
                        'SELECT sale_price FROM tbl_laboratory_item WHERE id = ?',
                        [item.lab_item_id]
                    );
                    if (labItems.length === 0) {
                        throw new Error(`Laboratory item ${item.lab_item_id} not found`);
                    }
                    itemPrice = parseFloat(labItems[0].sale_price) || 0;
                }

                const quantity = parseInt(item.quantity) || 1;
                calculatedTotal += itemPrice * quantity;

                const itemSql = `INSERT INTO tbl_laboratory_package_item 
                                (package_id, lab_item_id, quantity, item_price)
                                VALUES (?, ?, ?, ?)`;
                
                await conn.query(itemSql, [
                    packageId,
                    item.lab_item_id,
                    quantity,
                    itemPrice
                ]);
            }

            // Update package total price if calculated
            if (!total_price || parseFloat(total_price) === 0) {
                await conn.query(
                    'UPDATE tbl_laboratory_package SET total_price = ? WHERE id = ?',
                    [calculatedTotal, packageId]
                );
            }

            return { packageId, totalPrice: calculatedTotal || parseFloat(total_price) || 0 };
        });

        res.status(201).json({
            success: true,
            message: 'Laboratory package created successfully',
            data: {
                id: result.packageId,
                package_name,
                package_code,
                total_price: result.totalPrice
            }
        });
    } catch (error) {
        console.error('Error creating laboratory package:', error);
        res.status(500).json({
            success: false,
            message: 'Error creating laboratory package: ' + error.message
        });
    }
});

/**
 * @desc    Update laboratory package
 * @route   PUT /api/laboratory-packages/:id
 * @access  Private
 */
router.put('/:id', async (req, res, next) => {
    try {
        const {
            package_name,
            package_code,
            description,
            total_price,
            is_active,
            items // Array of {lab_item_id, quantity, item_price}
        } = req.body;

        // Check if package exists
        const existing = await query('SELECT * FROM tbl_laboratory_package WHERE id = ?', [req.params.id]);
        if (existing.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Laboratory package not found'
            });
        }

        // Check if package code already exists (excluding current package)
        if (package_code) {
            const codeExists = await query(
                'SELECT id FROM tbl_laboratory_package WHERE package_code = ? AND id != ?',
                [package_code, req.params.id]
            );
            if (codeExists.length > 0) {
                return res.status(400).json({
                    success: false,
                    message: 'Package code already exists'
                });
            }
        }

        // Use transaction
        const result = await transaction(async (conn) => {
            // Update package
            const updateSql = `UPDATE tbl_laboratory_package 
                              SET package_name = ?, package_code = ?, description = ?, 
                                  total_price = ?, is_active = ?
                              WHERE id = ?`;
            
            await conn.query(updateSql, [
                package_name || existing[0].package_name,
                package_code !== undefined ? package_code : existing[0].package_code,
                description !== undefined ? description : existing[0].description,
                total_price !== undefined ? parseFloat(total_price) : existing[0].total_price,
                is_active !== undefined ? (is_active ? 1 : 0) : existing[0].is_active,
                req.params.id
            ]);

            // Update items if provided
            if (items && Array.isArray(items)) {
                // Delete existing items
                await conn.query('DELETE FROM tbl_laboratory_package_item WHERE package_id = ?', [req.params.id]);

                // Calculate total
                let calculatedTotal = 0;

                // Insert new items
                for (const item of items) {
                    if (!item.lab_item_id) continue;

                    let itemPrice = parseFloat(item.item_price) || 0;
                    if (!itemPrice) {
                        const [labItems] = await conn.query(
                            'SELECT sale_price FROM tbl_laboratory_item WHERE id = ?',
                            [item.lab_item_id]
                        );
                        if (labItems.length > 0) {
                            itemPrice = parseFloat(labItems[0].sale_price) || 0;
                        }
                    }

                    const quantity = parseInt(item.quantity) || 1;
                    calculatedTotal += itemPrice * quantity;

                    const itemSql = `INSERT INTO tbl_laboratory_package_item 
                                    (package_id, lab_item_id, quantity, item_price)
                                    VALUES (?, ?, ?, ?)`;
                    
                    await conn.query(itemSql, [
                        req.params.id,
                        item.lab_item_id,
                        quantity,
                        itemPrice
                    ]);
                }

                // Update total price
                await conn.query(
                    'UPDATE tbl_laboratory_package SET total_price = ? WHERE id = ?',
                    [calculatedTotal, req.params.id]
                );
            }

            return { success: true };
        });

        res.json({
            success: true,
            message: 'Laboratory package updated successfully'
        });
    } catch (error) {
        console.error('Error updating laboratory package:', error);
        res.status(500).json({
            success: false,
            message: 'Error updating laboratory package: ' + error.message
        });
    }
});

/**
 * @desc    Delete laboratory package
 * @route   DELETE /api/laboratory-packages/:id
 * @access  Private
 */
router.delete('/:id', async (req, res, next) => {
    try {
        const packages = await query('SELECT package_name FROM tbl_laboratory_package WHERE id = ?', [req.params.id]);
        
        if (packages.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Laboratory package not found'
            });
        }

        // Delete package (cascade will delete items)
        await query('DELETE FROM tbl_laboratory_package WHERE id = ?', [req.params.id]);

        res.json({
            success: true,
            message: 'Laboratory package deleted successfully'
        });
    } catch (error) {
        console.error('Error deleting laboratory package:', error);
        res.status(500).json({
            success: false,
            message: 'Error deleting laboratory package: ' + error.message
        });
    }
});

/**
 * @desc    Get package items for POS (simplified)
 * @route   GET /api/laboratory-packages/:id/items
 * @access  Private
 */
router.get('/:id/items', async (req, res, next) => {
    try {
        const sql = `SELECT lpi.lab_item_id, lpi.quantity, lpi.item_price,
                     li.item_name, li.barcode, li.sale_price as current_price,
                     li.unit, li.reference_ranges
                     FROM tbl_laboratory_package_item lpi
                     JOIN tbl_laboratory_item li ON lpi.lab_item_id = li.id
                     WHERE lpi.package_id = ?
                     ORDER BY li.item_name`;
        
        const items = await query(sql, [req.params.id]);

        res.json({
            success: true,
            data: items
        });
    } catch (error) {
        console.error('Error fetching package items:', error);
        res.status(500).json({
            success: false,
            message: 'Error fetching package items: ' + error.message
        });
    }
});

module.exports = router;
