/**
 * Advanced Pharmacy Routes
 * Drug Formulary, Prescriptions, Medication Administration, Controlled Substances
 */

const express = require('express');
const router = express.Router();
const { protect } = require('../middleware/auth');
const { requireCrudPermission } = require('../middleware/permissions');
const {
    getDrugFormulary,
    addDrugToFormulary,
    checkDrugInteractions,
    createPrescription,
    getPrescriptions,
    dispenseMedication,
    getMedicationSchedule,
    administerMedication,
    logControlledSubstance,
    getControlledSubstanceAudit,
    reportAdverseDrugEvent,
    getAdverseDrugEvents,
    getPharmacyStats
} = require('../controllers/pharmacy.controller');

router.use(protect);
router.use(requireCrudPermission('pharmacy'));

// Drug Formulary
router.get('/formulary', getDrugFormulary);
router.post('/formulary', addDrugToFormulary);
router.post('/check-interactions', checkDrugInteractions);

// Prescriptions
router.post('/prescriptions', createPrescription);
router.get('/prescriptions', getPrescriptions);
router.patch('/prescriptions/:detailId/dispense', dispenseMedication);

// Medication Administration Record
router.get('/administration/patient/:patientId', getMedicationSchedule);
router.patch('/administration/:id/administer', administerMedication);

// Controlled Substances
router.post('/controlled-substances/log', logControlledSubstance);
router.get('/controlled-substances/audit', getControlledSubstanceAudit);

// Adverse Drug Events
router.post('/adverse-events', reportAdverseDrugEvent);
router.get('/adverse-events', getAdverseDrugEvents);

// Statistics
router.get('/stats', getPharmacyStats);

module.exports = router;

