const express = require('express');
const router = express.Router();
const { protect } = require('../middleware/auth');
const { requirePermission } = require('../middleware/permissions');
const { query } = require('../config/database');

router.use(protect);

// Normalize any date-like value to YYYY-MM-DD for MySQL DATE columns
const normalizeDateOnly = (value) => {
    if (!value) return null;
    if (value instanceof Date && !isNaN(value.getTime())) {
        return value.toISOString().split('T')[0];
    }
    const s = String(value);
    return s.length >= 10 ? s.substring(0, 10) : null;
};

// Get all purchases
router.get('/', async (req, res, next) => {
    try {
        const sql = `SELECT 
                        p.*,
                        pv.supplier_name as voucher_supplier_name,
                        pv.total_cost as voucher_total,
                        s.name as supplier_name,
                        s.id as supplier_id
                     FROM tbl_purchase p
                     LEFT JOIN tbl_purchase_voucher pv ON p.purchase_voucher_id = pv.id
                     LEFT JOIN tbl_supplier s ON p.supplier_id = s.id
                     ORDER BY p.date DESC 
                     LIMIT 200`;
        
        const purchases = await query(sql);
        
        res.json({ 
            success: true, 
            data: purchases,
            total: purchases.length,
            count: purchases.length
        });
    } catch (error) {
        console.error('Error fetching purchases:', error);
        res.json({ 
            success: true, 
            data: [],
            total: 0,
            count: 0,
            message: 'No purchases found'
        });
    }
});

// Get single purchase
router.get('/:id', async (req, res, next) => {
    try {
        const sql = `SELECT 
                        p.*,
                        pv.supplier_name as voucher_supplier_name,
                        pv.total_cost as voucher_total,
                        c.name as category_name,
                        s.name as supplier_name,
                        s.id as supplier_id
                     FROM tbl_purchase p
                     LEFT JOIN tbl_purchase_voucher pv ON p.purchase_voucher_id = pv.id
                     LEFT JOIN tbl_catalog c ON p.cat_id = c.id
                     LEFT JOIN tbl_supplier s ON p.supplier_id = s.id
                     WHERE p.id = ?`;
        
        const purchases = await query(sql, [req.params.id]);
        
        if (purchases.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Purchase not found'
            });
        }

        res.json({
            success: true,
            data: purchases[0]
        });
    } catch (error) {
        console.error('Error fetching purchase:', error);
        res.status(500).json({
            success: false,
            message: 'Error fetching purchase details'
        });
    }
});

// Create new purchase
router.post('/', requirePermission('purchases', 'create'), async (req, res, next) => {
    try {
        const {
            purchase_voucher_id,
            cat_id,
            cat_id_2,
            barcode,
            name,
            qty,
            qty_alert,
            unit_name,
            cost_price,
            total_cost_price,
            sale_price_latli,
            sale_price_latkar,
            remark,
            date,
            expire_date,
            alert_month,
            margin,
            warehouse_id,
            supplier_id
        } = req.body;

        // Debug logging
        console.log('Creating purchase with supplier_id:', supplier_id);

        const normalizedExpireDate = normalizeDateOnly(expire_date);

        // Calculate margin automatically if not provided
        let calculatedMargin = margin;
        if (!calculatedMargin && cost_price && sale_price_latli) {
            calculatedMargin = ((sale_price_latli - cost_price) / cost_price) * 100;
        }
        
        // Calculate alert_date from expire_date and alert_month
        let calculatedAlertDate = null;
        if (normalizedExpireDate && alert_month) {
            const expiry = new Date(normalizedExpireDate);
            if (!isNaN(expiry.getTime())) {
                const alertDate = new Date(expiry);
                alertDate.setMonth(alertDate.getMonth() - alert_month);
                calculatedAlertDate = alertDate.toISOString().split('T')[0];
            }
        }
        
        const sql = `INSERT INTO tbl_purchase 
                     (purchase_voucher_id, cat_id, cat_id_2, barcode, name, qty, qty_alert, 
                      unit_name, cost_price, total_cost_price, sale_price_latli, sale_price_latkar,
                      remark, date, expire_date, alert_month, alert_date, margin, status, supplier_id)
                     VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 1, ?)`;

        // Helper function to convert empty/undefined to NULL (avoid FK/strict-mode issues)
        const toNull = (value) => (value === undefined || value === '') ? null : value;
        
        const result = await query(sql, [
            toNull(purchase_voucher_id),
            toNull(cat_id),
            toNull(cat_id_2),
            toNull(barcode),
            toNull(name),
            qty || 0,
            qty_alert || 10,
            toNull(unit_name),
            cost_price || 0,
            total_cost_price || (cost_price * qty) || 0,
            sale_price_latli || 0,
            sale_price_latkar || 0,
            toNull(remark),
            date || new Date().toISOString().split('T')[0],
            toNull(normalizedExpireDate),
            toNull(alert_month),
            toNull(calculatedAlertDate),
            toNull(calculatedMargin),
            toNull(supplier_id)
        ]);

        console.log('Purchase created with ID:', result.insertId, 'supplier_id:', supplier_id);

        // Update stock automatically - ALWAYS add to POS inventory
        if (name && (qty || 0) > 0) {
            try {
                let stockUpdated = false;
                let stockId = null; // Initialize stockId variable
                
                // If barcode exists, try to find existing stock by barcode
                if (barcode) {
                    const barcodeStock = await query('SELECT id, qty FROM tbl_stock WHERE barcode = ?', [barcode]);
                    if (barcodeStock.length > 0) {
                        // Update existing stock by barcode
                        await query('UPDATE tbl_stock SET qty = qty + ?, cost_price = ?, sale_price_latli = COALESCE(?, sale_price_latli) WHERE barcode = ?', 
                                   [qty || 0, cost_price || 0, sale_price_latli, barcode]);
                        stockUpdated = true;
                    }
                }
                
                // If not updated by barcode, try to find by name
                if (!stockUpdated) {
                    const nameStock = await query('SELECT id, qty FROM tbl_stock WHERE name = ?', [name]);
                    if (nameStock.length > 0) {
                        // Update existing stock by name, also update barcode if provided
                        await query('UPDATE tbl_stock SET qty = qty + ?, cost_price = ?, sale_price_latli = COALESCE(?, sale_price_latli), barcode = COALESCE(?, barcode) WHERE name = ?', 
                                   [qty || 0, cost_price || 0, sale_price_latli, barcode, name]);
                        stockUpdated = true;
                    }
                }
                
                // If still not updated, create new stock entry
                if (!stockUpdated) {
                    // Generate barcode if not provided
                    let finalBarcode = barcode;
                    if (!finalBarcode) {
                        // Generate simple barcode based on name
                        const nameCode = name.replace(/[^a-zA-Z0-9]/g, '').toUpperCase();
                        const timestamp = Date.now().toString().slice(-6);
                        finalBarcode = `${nameCode.slice(0, 6)}${timestamp}`;
                    }
                    
                    // Calculate alert_date for stock
                    let stockAlertDate = null;
                    if (normalizedExpireDate && alert_month) {
                        const expiry = new Date(normalizedExpireDate);
                        if (!isNaN(expiry.getTime())) {
                            const alertDate = new Date(expiry);
                            alertDate.setMonth(alertDate.getMonth() - alert_month);
                            stockAlertDate = alertDate.toISOString().split('T')[0];
                        }
                    }
                    
                    // Calculate margin for stock
                    const stockSalePrice = sale_price_latli || cost_price || 0;
                    const stockMargin = cost_price && stockSalePrice ? 
                        ((stockSalePrice - cost_price) / cost_price) * 100 : 
                        calculatedMargin;
                    
                    await query(`INSERT INTO tbl_stock 
                                (barcode, name, cat_id, cat_id_2, qty, qty_alert, unit_name, 
                                 cost_price, sale_price_latli, sale_price_latkar, remark, expire_date, alert_month, alert_date, margin)
                                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
                               [finalBarcode, name, toNull(cat_id), toNull(cat_id_2), qty || 0, qty_alert || 10,
                                toNull(unit_name), cost_price || 0, stockSalePrice, 
                                sale_price_latkar || stockSalePrice,
                                toNull(remark), toNull(normalizedExpireDate), toNull(alert_month), toNull(stockAlertDate), toNull(stockMargin)]);
                    // Get the stock ID that was just created
                    const newStockResult = await query('SELECT id FROM tbl_stock WHERE barcode = ? OR name = ? ORDER BY id DESC LIMIT 1', [finalBarcode, name]);
                    if (newStockResult.length > 0) {
                        stockId = newStockResult[0].id;
                    }
                }
                
                // Get the stock_id that was just created/updated
                if (!stockId) {
                    if (stockUpdated) {
                        // Find the stock ID that was updated
                        if (barcode) {
                            const stockResult = await query('SELECT id FROM tbl_stock WHERE barcode = ?', [barcode]);
                            if (stockResult.length > 0) {
                                stockId = stockResult[0].id;
                            }
                        }
                        if (!stockId) {
                            const stockResult = await query('SELECT id FROM tbl_stock WHERE name = ?', [name]);
                            if (stockResult.length > 0) {
                                stockId = stockResult[0].id;
                            }
                        }
                    }
                }
                
                // If warehouse_id is provided, create stock movement record
                // The database trigger will automatically update warehouse stock
                if (warehouse_id && stockId && (qty || 0) > 0) {
                    try {
                        // Create stock movement record for warehouse
                        // The trigger trg_update_warehouse_stock_on_movement will automatically
                        // update tbl_warehouse_stock when this record is inserted
                        // Add reference_id to link movement to purchase
                        await query(
                            `INSERT INTO tbl_stock_movement
                             (movement_type, warehouse_id, stock_id, quantity, reference_type, reference_id,
                              movement_date, notes, created_by)
                             VALUES ('IN', ?, ?, ?, 'PURCHASE', ?, NOW(), ?, ?)`,
                            [
                                warehouse_id,
                                stockId,
                                qty || 0,
                                result.insertId, // Link to purchase record
                                `Purchase: ${name}`,
                                req.user?.id || null
                            ]
                        );
                        
                        console.log(`✅ Warehouse stock movement created for purchase: ${name} (${qty} units) in warehouse ${warehouse_id}`);
                    } catch (warehouseError) {
                        console.error('Warehouse stock movement error:', warehouseError.message);
                        // Don't throw error, just log it so purchase still succeeds
                    }
                }
                
                console.log(`✅ Stock updated for purchase: ${name} (${qty} units)`);
            } catch (stockError) {
                console.error('Stock update error:', stockError.message);
                // Don't throw error, just log it so purchase still succeeds
            }
        }

        res.status(201).json({
            success: true,
            message: 'Purchase created successfully',
            data: {
                id: result.insertId,
                ...req.body
            }
        });
    } catch (error) {
        next(error);
    }
});

// Update purchase
router.put('/:id', requirePermission('purchases', 'update'), async (req, res, next) => {
    try {
        const {
            purchase_voucher_id,
            cat_id,
            cat_id_2,
            barcode,
            name,
            qty,
            qty_alert,
            unit_name,
            cost_price,
            total_cost_price,
            sale_price_latli,
            sale_price_latkar,
            remark,
            date,
            expire_date,
            alert_month,
            margin,
            supplier_id
        } = req.body;

        const normalizedExpireDate = normalizeDateOnly(expire_date);

        // Get old purchase data for stock adjustment
        const oldPurchase = await query('SELECT qty, barcode FROM tbl_purchase WHERE id = ?', [req.params.id]);
        
        if (oldPurchase.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Purchase not found'
            });
        }

        // Calculate margin automatically if not provided
        let calculatedMargin = margin;
        if (!calculatedMargin && cost_price && sale_price_latli) {
            calculatedMargin = ((sale_price_latli - cost_price) / cost_price) * 100;
        }
        
        // Calculate alert_date from expire_date and alert_month
        let calculatedAlertDate = null;
        if (normalizedExpireDate && alert_month) {
            const expiry = new Date(normalizedExpireDate);
            if (!isNaN(expiry.getTime())) {
                const alertDate = new Date(expiry);
                alertDate.setMonth(alertDate.getMonth() - alert_month);
                calculatedAlertDate = alertDate.toISOString().split('T')[0];
            }
        }
        
        const sql = `UPDATE tbl_purchase 
                     SET purchase_voucher_id = ?, cat_id = ?, cat_id_2 = ?, barcode = ?, name = ?, 
                         qty = ?, qty_alert = ?, unit_name = ?, cost_price = ?, total_cost_price = ?, 
                         sale_price_latli = ?, sale_price_latkar = ?, remark = ?, date = ?, 
                         expire_date = ?, alert_month = ?, alert_date = ?, margin = ?, supplier_id = ?
                     WHERE id = ?`;

        // Helper function to convert empty/undefined to NULL (avoid FK/strict-mode issues)
        const toNull = (value) => (value === undefined || value === '') ? null : value;
        
        const result = await query(sql, [
            toNull(purchase_voucher_id),
            toNull(cat_id),
            toNull(cat_id_2),
            toNull(barcode),
            toNull(name),
            qty || 0,
            qty_alert || 10,
            toNull(unit_name),
            cost_price || 0,
            total_cost_price || (cost_price * qty) || 0,
            sale_price_latli || 0,
            sale_price_latkar || 0,
            toNull(remark),
            date || new Date().toISOString().split('T')[0],
            toNull(normalizedExpireDate),
            toNull(alert_month),
            toNull(calculatedAlertDate),
            toNull(calculatedMargin),
            toNull(supplier_id),
            req.params.id
        ]);

        // Smart stock adjustment for purchase edit
        try {
            const oldQty = oldPurchase[0].qty || 0;
            const oldBarcode = oldPurchase[0].barcode;
            const oldName = oldPurchase[0].name;
            const newQty = qty || 0;
            
            // Revert old stock first
            if (oldQty > 0) {
                if (oldBarcode) {
                    const oldStock = await query('SELECT id FROM tbl_stock WHERE barcode = ?', [oldBarcode]);
                    if (oldStock.length > 0) {
                        await query('UPDATE tbl_stock SET qty = qty - ? WHERE barcode = ?', [oldQty, oldBarcode]);
                    }
                } else if (oldName) {
                    const oldStock = await query('SELECT id FROM tbl_stock WHERE name = ?', [oldName]);
                    if (oldStock.length > 0) {
                        await query('UPDATE tbl_stock SET qty = qty - ? WHERE name = ?', [oldQty, oldName]);
                    }
                }
            }
            
            // Apply new stock using the same logic as create
            if (name && newQty > 0) {
                let stockUpdated = false;
                
                // Try to update by barcode first
                if (barcode) {
                    const barcodeStock = await query('SELECT id, qty FROM tbl_stock WHERE barcode = ?', [barcode]);
                    if (barcodeStock.length > 0) {
                        await query('UPDATE tbl_stock SET qty = qty + ?, cost_price = ?, sale_price_latli = COALESCE(?, sale_price_latli) WHERE barcode = ?', 
                                   [newQty, cost_price || 0, sale_price_latli, barcode]);
                        stockUpdated = true;
                    }
                }
                
                // Try to update by name if not updated by barcode
                if (!stockUpdated) {
                    const nameStock = await query('SELECT id, qty FROM tbl_stock WHERE name = ?', [name]);
                    if (nameStock.length > 0) {
                        await query('UPDATE tbl_stock SET qty = qty + ?, cost_price = ?, sale_price_latli = COALESCE(?, sale_price_latli), barcode = COALESCE(?, barcode) WHERE name = ?', 
                                   [newQty, cost_price || 0, sale_price_latli, barcode, name]);
                        stockUpdated = true;
                    }
                }
                
                // Create new stock if still not found
                if (!stockUpdated) {
                    let finalBarcode = barcode;
                    if (!finalBarcode) {
                        const nameCode = name.replace(/[^a-zA-Z0-9]/g, '').toUpperCase();
                        const timestamp = Date.now().toString().slice(-6);
                        finalBarcode = `${nameCode.slice(0, 6)}${timestamp}`;
                    }
                    
                    await query(`INSERT INTO tbl_stock 
                                (barcode, name, cat_id, cat_id_2, qty, qty_alert, unit_name, 
                                 cost_price, sale_price_latli, sale_price_latkar, remark, expire_date, alert_month, margin)
                                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
                               [finalBarcode, name, toNull(cat_id), toNull(cat_id_2), newQty, qty_alert || 10,
                                toNull(unit_name), cost_price || 0, sale_price_latli || cost_price || 0, 
                                sale_price_latkar || sale_price_latli || cost_price || 0,
                                toNull(remark), toNull(normalizedExpireDate), toNull(alert_month), toNull(margin)]);
                }
                
                console.log(`✅ Stock adjusted for purchase edit: ${name} (${newQty} units)`);
            }
        } catch (stockError) {
            console.error('Stock adjustment error:', stockError.message);
        }

        if (result.affectedRows === 0) {
            return res.status(404).json({
                success: false,
                message: 'Purchase not found'
            });
        }

        res.json({
            success: true,
            message: 'Purchase updated successfully',
            data: {
                id: req.params.id,
                affectedRows: result.affectedRows
            }
        });
    } catch (error) {
        next(error);
    }
});

// Delete purchase
router.delete('/:id', requirePermission('purchases', 'delete'), async (req, res, next) => {
    try {
        // Get purchase data for stock adjustment
        const purchase = await query('SELECT qty, barcode, name FROM tbl_purchase WHERE id = ?', [req.params.id]);
        
        if (purchase.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Purchase not found'
            });
        }

        const { qty, barcode, name } = purchase[0];

        // Delete purchase
        const result = await query('DELETE FROM tbl_purchase WHERE id = ?', [req.params.id]);

        // Adjust stock (subtract the purchased quantity) - try barcode first, then name
        if (qty > 0) {
            try {
                let stockAdjusted = false;
                
                // Try to adjust by barcode first
                if (barcode) {
                    const barcodeStock = await query('SELECT id FROM tbl_stock WHERE barcode = ?', [barcode]);
                    if (barcodeStock.length > 0) {
                        await query('UPDATE tbl_stock SET qty = qty - ? WHERE barcode = ?', [qty, barcode]);
                        stockAdjusted = true;
                        console.log(`✅ Stock reduced by barcode for deleted purchase: ${name} (-${qty} units)`);
                    }
                }
                
                // If not adjusted by barcode, try by name
                if (!stockAdjusted && name) {
                    const nameStock = await query('SELECT id FROM tbl_stock WHERE name = ?', [name]);
                    if (nameStock.length > 0) {
                        await query('UPDATE tbl_stock SET qty = qty - ? WHERE name = ?', [qty, name]);
                        console.log(`✅ Stock reduced by name for deleted purchase: ${name} (-${qty} units)`);
                    }
                }
            } catch (stockError) {
                console.error('Stock adjustment error:', stockError.message);
            }
        }

        res.json({
            success: true,
            message: 'Purchase deleted successfully',
            data: {
                id: req.params.id,
                affectedRows: result.affectedRows
            }
        });
    } catch (error) {
        next(error);
    }
});

module.exports = router;

