const express = require('express');
const router = express.Router();
const { protect } = require('../middleware/auth');
const { requireCrudPermission } = require('../middleware/permissions');
const { query } = require('../config/database');

router.use(protect);
router.use(requireCrudPermission('reports'));
// Sales by voucher (with doctor/patient if linked via EMR)
router.get('/sales-by-voucher', async (req, res, next) => {
    try {
        const { from, to, search } = req.query;
        console.log('📊 Sales by Voucher Request - Date Range:', { from, to, search });
        
        const params = [];
        const params2 = []; // Separate params for inpatient sales
        let where = '1=1';
        let whereInpatient = '1=1';
        
        // Handle date filtering - ensure we cover the entire day
        if (from) { 
            where += ' AND DATE(v.Date) >= ?'; 
            params.push(from);
            whereInpatient += ' AND DATE(s.Date) >= ?';
            params2.push(from);
        }
        if (to) { 
            where += ' AND DATE(v.Date) <= ?'; 
            params.push(to);
            whereInpatient += ' AND DATE(s.Date) <= ?';
            params2.push(to);
        }

        // Handle search filtering for voucher sales
        if (search && search.trim()) {
            const searchTerm = `%${search.trim()}%`;
            where += ` AND (
                CAST(v.id AS CHAR) LIKE ? OR
                COALESCE(v.doctor_name, '') LIKE ? OR
                COALESCE(p.name, '') LIKE ? OR
                COALESCE(v.customer_name, '') LIKE ?
            )`;
            params.push(searchTerm, searchTerm, searchTerm, searchTerm);
        }

        // Handle search filtering for inpatient sales
        if (search && search.trim()) {
            const searchTerm = `%${search.trim()}%`;
            whereInpatient += ` AND (
                CAST(s.VNo AS CHAR) LIKE ? OR
                COALESCE(s.Cashier, '') LIKE ?
            )`;
            params2.push(searchTerm, searchTerm);
        }

        const sql = `
            SELECT 
                v.id AS voucher_no,
                DATE_FORMAT(v.Date, '%Y-%m-%d %H:%i') AS date_time,
                COALESCE(v.doctor_name, '') AS doctor_name,
                COALESCE(p.name, v.customer_name, 'Walk-in') AS patient_name,
                COALESCE(v.sub_total, 0) as sub_total,
                COALESCE(v.actual_cost, 0) AS total,
                COALESCE(v.dis, 0) as dis,
                COALESCE(v.payment, 'Cash') as payment,
                COUNT(s.id) AS items
            FROM tbl_voucher v
            LEFT JOIN tbl_sale s ON s.VNo = v.id
            LEFT JOIN tbl_patient p ON v.patient_id = p.id
            WHERE ${where}
            GROUP BY v.id, v.Date, v.doctor_name, v.customer_name, p.name, v.sub_total, v.actual_cost, v.dis, v.payment
            
            UNION ALL
            
            SELECT 
                s.VNo AS voucher_no,
                DATE_FORMAT(s.Date, '%Y-%m-%d %H:%i') AS date_time,
                'Inpatient' AS doctor_name,
                COALESCE(s.Cashier, 'Inpatient Sale') AS patient_name,
                SUM(s.Total) as sub_total,
                SUM(s.Total) AS total,
                0 as dis,
                'Cash' as payment,
                COUNT(s.id) AS items
            FROM tbl_sale s
            WHERE s.SaleType = 'Inpatient' 
            AND s.Barcode NOT LIKE 'BED-%' 
            AND s.Barcode NOT LIKE 'DISC-%' 
            AND s.Barcode NOT LIKE 'TAX-%'
            AND ${whereInpatient}
            GROUP BY s.VNo, s.Date, s.Cashier
            
            ORDER BY date_time DESC`;

        // Combine both parameter arrays for the UNION query
        const allParams = [...params, ...params2];
        console.log('📊 Executing SQL with params:', allParams);
        const rows = await query(sql, allParams);
        console.log(`📊 Found ${rows ? rows.length : 0} vouchers`);
        
        res.json({ success: true, data: rows || [] });
    } catch (error) {
        console.error('❌ Sales by voucher error:', error);
        next(error);
    }
});


// Sales report
router.get('/sales', async (req, res, next) => {
    try {
        const { startDate, endDate } = req.query;
        const sql = `SELECT Date, SUM(Total) as total_sales, SUM(Profit) as total_profit, COUNT(DISTINCT VNo) as total_vouchers
                     FROM tbl_sale 
                     WHERE Date BETWEEN ? AND ?
                     GROUP BY Date
                     ORDER BY Date DESC`;
        const report = await query(sql, [startDate, endDate]);
        res.json({ success: true, data: report });
    } catch (error) {
        next(error);
    }
});

// Profit report
router.get('/profit', async (req, res, next) => {
    try {
        const { startDate, endDate } = req.query;
        const sql = `SELECT SUM(Profit) as total_profit, SUM(Total) as total_sales
                     FROM tbl_sale 
                     WHERE Date BETWEEN ? AND ?`;
        const result = await query(sql, [startDate, endDate]);
        res.json({ success: true, data: result[0] });
    } catch (error) {
        next(error);
    }
});

// Stock Sales Report (Products only, not services)
router.get('/stock-sales', async (req, res, next) => {
    try {
        const { startDate, endDate, search } = req.query;
        
        console.log('Stock Sales Request:', { startDate, endDate, search });
        
        // Build query for stock/product sales (exclude services and returned items)
        let sql = `
            SELECT 
                s.id,
                s.VNo as voucher_no,
                s.VNo as invoice_no,
                DATE_FORMAT(s.Date, '%Y-%m-%d') as sale_date,
                s.Name as product_name,
                s.Barcode as barcode,
                COALESCE(s.SaleType, 'product') as sale_type,
                s.SalePrice as unit_price,
                s.Qty as quantity,
                COALESCE(s.dis, 0) as discount,
                s.Total as total,
                COALESCE(s.Profit, 0) as profit,
                COALESCE(s.cost_price, 0) as cost_price,
                COALESCE(c.name, '-') as category,
                COALESCE(s.Cashier, '-') as cashier
            FROM tbl_sale s
            LEFT JOIN tbl_catalog c ON s.cat_id = c.id
            WHERE 1=1
        `;
        
        const params = [];
        
        // Filter for products only (include regular products and inpatient products, exclude services and special inpatient entries)
        sql += ` AND (
                    (s.SaleType != 'service' OR s.SaleType IS NULL) AND (s.Barcode NOT LIKE 'S%' OR s.Barcode IS NULL)
                    OR (s.SaleType = 'Inpatient' AND s.Barcode NOT LIKE 'BED-%' AND s.Barcode NOT LIKE 'DISC-%' AND s.Barcode NOT LIKE 'TAX-%')
                 )`;
        
        // Exclude items that have been fully returned
        // This checks if the total returned quantity for this barcode in this voucher equals or exceeds the original sale quantity
        sql += ` AND NOT EXISTS (
                    SELECT 1
                    FROM tbl_sale_return_item sri
                    INNER JOIN tbl_voucher_return vr ON sri.vno = vr.id
                    WHERE vr.customer_name = CONCAT('Return for Voucher ', s.VNo)
                    AND sri.barcode = s.Barcode
                    GROUP BY sri.barcode
                    HAVING SUM(sri.qty) >= s.Qty
                 )`;
        
        // Date filtering
        if (startDate && endDate) {
            sql += ' AND s.Date BETWEEN ? AND ?';
            params.push(startDate, endDate);
        } else if (startDate) {
            sql += ' AND s.Date >= ?';
            params.push(startDate);
        } else if (endDate) {
            sql += ' AND s.Date <= ?';
            params.push(endDate);
        }
        
        // Search filtering
        if (search && search.trim()) {
            sql += ` AND (
                s.Name LIKE ? OR 
                s.Barcode LIKE ? OR
                c.name LIKE ?
            )`;
            const searchTerm = `%${search.trim()}%`;
            params.push(searchTerm, searchTerm, searchTerm);
        }
        
        sql += ' ORDER BY s.Date DESC, s.VNo DESC LIMIT 500';
        
        console.log('Executing Stock Sales SQL with params:', params);
        console.log('Stock sales SQL:', sql);
        const stocks = await query(sql, params);
        console.log(`Found ${stocks ? stocks.length : 0} stock sales records`);
        
        // Debug: Show sample results
        if (stocks && stocks.length > 0) {
            console.log('Sample stock sales results:', stocks.slice(0, 3).map(r => ({
                name: r.product_name,
                barcode: r.barcode,
                sale_type: r.sale_type,
                total: r.total
            })));
        }
        
        // Handle case where no results are returned
        const safeStocks = stocks || [];
        
        // Calculate summary statistics
        const summaryData = safeStocks.reduce((acc, item) => {
            acc.totalAmount += parseFloat(item.total) || 0;
            acc.totalProfit += parseFloat(item.profit) || 0;
            acc.totalDiscount += parseFloat(item.discount) || 0;
            acc.totalQuantity += parseInt(item.quantity) || 0;
            return acc;
        }, {
            totalAmount: 0,
            totalProfit: 0,
            totalDiscount: 0,
            totalQuantity: 0,
            totalRecords: safeStocks.length
        });
        
        res.json({ 
            success: true, 
            data: safeStocks,
            summary: summaryData
        });
    } catch (error) {
        console.error('❌ Stock Sales Error:', error.message);
        console.error('Error code:', error.code);
        console.error('Error SQL state:', error.sqlState);
        console.error('Error stack:', error.stack);
        console.error('Full error object:', JSON.stringify(error, Object.getOwnPropertyNames(error)));
        next(error);
    }
});

// Service Sales Report with Doctor and Patient Information
router.get('/service-sales', async (req, res, next) => {
    try {
        const { startDate, endDate, search } = req.query;
        
        console.log('Service Sales Request:', { startDate, endDate, search });
        
        // Build simplified query (removed service catalog join to avoid potential issues)
        let sql = `
            SELECT 
                s.id,
                s.VNo as voucher_no,
                DATE_FORMAT(s.Date, '%Y-%m-%d') as sale_date,
                DATE_FORMAT(COALESCE(v.Date, s.Date), '%Y-%m-%d %H:%i') as date_time,
                s.Name as service_name,
                s.Barcode as barcode,
                COALESCE(s.SaleType, 'product') as sale_type,
                s.SalePrice as unit_price,
                s.Qty as quantity,
                COALESCE(s.dis, 0) as discount,
                s.Total as total,
                COALESCE(s.Profit, 0) as profit,
                COALESCE(s.cost_price, 0) as cost_price,
                COALESCE(v.doctor_name, '-') as doctor_name,
                COALESCE(v.doctor_id, 0) as doctor_id,
                COALESCE(p.name, COALESCE(v.customer_name, 'Walk-in')) as patient_name,
                COALESCE(p.id, 0) as patient_id,
                COALESCE(p.phone, '-') as patient_phone,
                COALESCE(p.address, '-') as patient_address,
                COALESCE(v.payment, 'Cash') as payment,
                COALESCE(s.Cashier, '-') as cashier,
                COALESCE(sc.name, '-') as service_category
            FROM tbl_sale s
            LEFT JOIN tbl_voucher v ON s.VNo = v.id
            LEFT JOIN tbl_patient p ON v.patient_id = p.id
            LEFT JOIN tbl_service srv ON s.Barcode = srv.barcode
            LEFT JOIN tbl_service_catalog sc ON srv.cat_id = sc.id
            WHERE 1=1
        `;
        
        const params = [];
        
        // Filter for services only (SaleType = 'service' OR Barcode starts with 'S', or inpatient services, but exclude special inpatient entries)
        sql += ` AND ((s.SaleType = 'service' OR s.Barcode LIKE 'S%') 
                      OR (s.SaleType = 'Inpatient' AND s.Barcode LIKE 'INP-%'))
                 AND s.Barcode NOT LIKE 'BED-%' AND s.Barcode NOT LIKE 'DISC-%' AND s.Barcode NOT LIKE 'TAX-%'`;
        
        // Date filtering
        if (startDate && endDate) {
            sql += ' AND s.Date BETWEEN ? AND ?';
            params.push(startDate, endDate);
        } else if (startDate) {
            sql += ' AND s.Date >= ?';
            params.push(startDate);
        } else if (endDate) {
            sql += ' AND s.Date <= ?';
            params.push(endDate);
        }
        
        // Search filtering (service name, doctor name, or patient name)
        if (search && search.trim()) {
            sql += ` AND (
                s.Name LIKE ? OR 
                COALESCE(v.doctor_name, '') LIKE ? OR 
                COALESCE(p.name, '') LIKE ? OR 
                COALESCE(v.customer_name, '') LIKE ? OR
                s.Barcode LIKE ?
            )`;
            const searchTerm = `%${search.trim()}%`;
            params.push(searchTerm, searchTerm, searchTerm, searchTerm, searchTerm);
        }
        
        sql += ' ORDER BY s.Date DESC, s.VNo DESC LIMIT 500';
        
        console.log('Executing SQL with params:', params);
        const services = await query(sql, params);
        console.log(`Found ${services ? services.length : 0} service sales records`);
        
        // Handle case where no results are returned
        const safeServices = services || [];
        
        // Calculate summary statistics
        const summaryData = safeServices.reduce((acc, item) => {
            acc.totalAmount += parseFloat(item.total) || 0;
            acc.totalProfit += parseFloat(item.profit) || 0;
            acc.totalDiscount += parseFloat(item.discount) || 0;
            acc.totalQuantity += parseInt(item.quantity) || 0;
            return acc;
        }, {
            totalAmount: 0,
            totalProfit: 0,
            totalDiscount: 0,
            totalQuantity: 0,
            totalRecords: safeServices.length
        });
        
        res.json({ 
            success: true, 
            data: safeServices,
            summary: summaryData
        });
    } catch (error) {
        console.error('Service Sales Error:', error.message);
        console.error('Error details:', error);
        next(error);
    }
});

// Service Sales Summary by Service
router.get('/service-sales-summary', async (req, res, next) => {
    try {
        const { startDate, endDate } = req.query;
        
        let sql = `
            SELECT 
                s.Name as service_name,
                s.Barcode as barcode,
                sc.name as category,
                COUNT(*) as sale_count,
                SUM(s.Qty) as total_quantity,
                SUM(s.Total) as total_sales,
                SUM(s.Profit) as total_profit,
                AVG(s.SalePrice) as avg_price,
                MIN(s.Date) as first_sale,
                MAX(s.Date) as last_sale
            FROM tbl_sale s
            LEFT JOIN tbl_service srv ON s.Barcode = srv.barcode
            LEFT JOIN tbl_service_catalog sc ON srv.cat_id = sc.id
            WHERE (s.SaleType = 'service' OR s.Barcode LIKE 'S%')
        `;
        
        const params = [];
        
        if (startDate && endDate) {
            sql += ' AND s.Date BETWEEN ? AND ?';
            params.push(startDate, endDate);
        }
        
        sql += ' GROUP BY s.Name, s.Barcode, sc.name ORDER BY total_sales DESC LIMIT 100';
        
        const summary = await query(sql, params);
        
        res.json({ success: true, data: summary });
    } catch (error) {
        next(error);
    }
});

// Service Sales by Doctor
router.get('/service-sales-by-doctor', async (req, res, next) => {
    try {
        const { startDate, endDate } = req.query;
        
        let sql = `
            SELECT 
                COALESCE(v.doctor_name, 'No Doctor Assigned') as doctor_name,
                v.doctor_id,
                COUNT(DISTINCT s.VNo) as total_vouchers,
                COUNT(s.id) as service_count,
                SUM(s.Qty) as total_quantity,
                SUM(s.Total) as total_sales,
                SUM(s.Profit) as total_profit
            FROM tbl_sale s
            LEFT JOIN tbl_voucher v ON s.VNo = v.id
            WHERE (s.SaleType = 'service' OR s.Barcode LIKE 'S%')
        `;
        
        const params = [];
        
        if (startDate && endDate) {
            sql += ' AND s.Date BETWEEN ? AND ?';
            params.push(startDate, endDate);
        }
        
        sql += ' GROUP BY v.doctor_name, v.doctor_id ORDER BY total_sales DESC';
        
        const doctorSales = await query(sql, params);
        
        res.json({ success: true, data: doctorSales });
    } catch (error) {
        next(error);
    }
});

// Doctor Performance Report
router.get('/doctor-performance', async (req, res, next) => {
    try {
        const { startDate, endDate, doctorId } = req.query;
        
        console.log('📊 Doctor Performance Report Request:', { startDate, endDate, doctorId });
        
        // Modified query to include BOTH treatment-based AND direct voucher sales
        let sql = `
            SELECT 
                d.id as doctor_id,
                d.name as doctor_name,
                d.specialist as specialization,
                d.phone as doctor_phone,
                DATE(COALESCE(t.Date, v.Date)) as treatment_date,
                COUNT(DISTINCT COALESCE(t.patient_id, v.patient_id)) as patient_count,
                COUNT(DISTINCT COALESCE(t.id, v.id)) as treatment_count,
                COALESCE(SUM(v.actual_cost), 0) as total_revenue,
                COALESCE(AVG(v.actual_cost), 0) as avg_revenue_per_patient,
                COALESCE(SUM(v.dis), 0) as total_discount,
                GROUP_CONCAT(DISTINCT t.Treatment_Type) as treatment_types,
                GROUP_CONCAT(DISTINCT CONCAT(v.id, ':', COALESCE(v.actual_cost, 0)) SEPARATOR '|') as invoice_details
            FROM tbl_doctor d
            LEFT JOIN tbl_treatment t ON d.id = t.doctor_id
            LEFT JOIN tbl_voucher v ON (t.Voucher_id = v.id OR (v.doctor_id = d.id AND t.id IS NULL))
            WHERE 1=1
        `;
        
        const params = [];
        
        // Specific doctor filter - apply this first to narrow down results
        if (doctorId) {
            sql += ' AND d.id = ?';
            params.push(doctorId);
        }
        
        // Ensure we only get doctors with actual treatments or vouchers
        sql += ' AND (t.id IS NOT NULL OR v.doctor_id IS NOT NULL)';
        
        // Date filtering - check both treatment date and voucher date
        if (startDate) {
            sql += ' AND DATE(COALESCE(t.Date, v.Date)) >= ?';
            params.push(startDate);
        }
        
        if (endDate) {
            sql += ' AND DATE(COALESCE(t.Date, v.Date)) <= ?';
            params.push(endDate);
        }
        
        sql += ` GROUP BY d.id, d.name, d.specialist, d.phone, DATE(COALESCE(t.Date, v.Date))
                 HAVING treatment_count > 0
                 ORDER BY treatment_date DESC, total_revenue DESC`;
        
        console.log('Executing Doctor Performance SQL with params:', params);
        const performance = await query(sql, params);
        console.log(`Found ${performance ? performance.length : 0} doctor performance records`);
        
        // Calculate summary
        const safePerformance = performance || [];
        const summary = safePerformance.reduce((acc, record) => {
            acc.totalPatients += parseInt(record.patient_count) || 0;
            acc.totalTreatments += parseInt(record.treatment_count) || 0;
            acc.totalRevenue += parseFloat(record.total_revenue) || 0;
            acc.totalDiscount += parseFloat(record.total_discount) || 0;
            return acc;
        }, {
            totalPatients: 0,
            totalTreatments: 0,
            totalRevenue: 0,
            totalDiscount: 0,
            totalRecords: safePerformance.length
        });
        
        res.json({ 
            success: true, 
            data: safePerformance,
            summary: summary
        });
    } catch (error) {
        console.error('❌ Doctor Performance Report Error:', error);
        next(error);
    }
});

// Doctor Summary Report (aggregated by doctor)
router.get('/doctor-summary', async (req, res, next) => {
    try {
        const { startDate, endDate, doctorId } = req.query;
        
        // Modified query to include BOTH treatment-based AND direct voucher sales
        let sql = `
            SELECT 
                d.id as doctor_id,
                d.name as doctor_name,
                d.specialist as specialization,
                d.phone as doctor_phone,
                COUNT(DISTINCT COALESCE(t.patient_id, v.patient_id)) as total_patients,
                COUNT(DISTINCT COALESCE(t.id, v.id)) as total_treatments,
                COUNT(DISTINCT DATE(COALESCE(t.Date, v.Date))) as days_worked,
                COALESCE(SUM(v.actual_cost), 0) as total_revenue,
                COALESCE(AVG(v.actual_cost), 0) as avg_revenue_per_patient,
                COALESCE(SUM(v.dis), 0) as total_discount,
                MIN(DATE(COALESCE(t.Date, v.Date))) as first_treatment_date,
                MAX(DATE(COALESCE(t.Date, v.Date))) as last_treatment_date
            FROM tbl_doctor d
            LEFT JOIN tbl_treatment t ON d.id = t.doctor_id
            LEFT JOIN tbl_voucher v ON (t.Voucher_id = v.id OR (v.doctor_id = d.id AND t.id IS NULL))
            WHERE 1=1
        `;
        
        const params = [];
        
        // Specific doctor filter - apply this first to narrow down results
        if (doctorId) {
            sql += ' AND d.id = ?';
            params.push(doctorId);
        }
        
        // Ensure we only get doctors with actual treatments or vouchers
        sql += ' AND (t.id IS NOT NULL OR v.doctor_id IS NOT NULL)';
        
        if (startDate) {
            sql += ' AND DATE(COALESCE(t.Date, v.Date)) >= ?';
            params.push(startDate);
        }
        
        if (endDate) {
            sql += ' AND DATE(COALESCE(t.Date, v.Date)) <= ?';
            params.push(endDate);
        }
        
        sql += ` GROUP BY d.id, d.name, d.specialist, d.phone
                 HAVING total_treatments > 0
                 ORDER BY total_revenue DESC`;
        
        const summary = await query(sql, params);
        
        res.json({ 
            success: true, 
            data: summary || []
        });
    } catch (error) {
        console.error('❌ Doctor Summary Report Error:', error);
        next(error);
    }
});

// Comprehensive Sales Report (includes all sales types)
router.get('/comprehensive-sales', async (req, res, next) => {
    try {
        const { startDate, endDate, saleType } = req.query;
        
        console.log('Comprehensive Sales Request:', { startDate, endDate, saleType });
        
        let sql = `
            SELECT 
                s.id,
                s.VNo as voucher_no,
                DATE_FORMAT(s.Date, '%Y-%m-%d') as sale_date,
                DATE_FORMAT(COALESCE(v.Date, s.Date), '%Y-%m-%d %H:%i') as date_time,
                s.Name as item_name,
                s.Barcode as barcode,
                COALESCE(s.SaleType, 'Retail') as sale_type,
                s.SalePrice as unit_price,
                s.Qty as quantity,
                COALESCE(s.dis, 0) as discount,
                s.Total as total,
                COALESCE(s.Profit, 0) as profit,
                COALESCE(s.cost_price, 0) as cost_price,
                COALESCE(c.name, '-') as category,
                COALESCE(v.doctor_name, '-') as doctor_name,
                COALESCE(v.doctor_id, 0) as doctor_id,
                COALESCE(p.name, COALESCE(v.customer_name, 'Walk-in')) as patient_name,
                COALESCE(p.id, 0) as patient_id,
                COALESCE(v.payment, 'Cash') as payment,
                COALESCE(s.Cashier, '-') as cashier,
                CASE 
                    WHEN s.SaleType = 'Inpatient' THEN 'Inpatient'
                    WHEN s.SaleType = 'service' OR s.Barcode LIKE 'S%' THEN 'Service'
                    ELSE 'Product'
                END as item_category
            FROM tbl_sale s
            LEFT JOIN tbl_voucher v ON s.VNo = v.id
            LEFT JOIN tbl_patient p ON v.patient_id = p.id
            LEFT JOIN tbl_catalog c ON s.cat_id = c.id
            WHERE 1=1
        `;
        
        const params = [];
        
        // Filter by sale type if specified
        if (saleType) {
            if (saleType === 'inpatient') {
                sql += ` AND s.SaleType = 'Inpatient'`;
            } else if (saleType === 'service') {
                sql += ` AND (s.SaleType = 'service' OR s.Barcode LIKE 'S%') 
                         AND s.Barcode NOT LIKE 'BED-%' AND s.Barcode NOT LIKE 'DISC-%' AND s.Barcode NOT LIKE 'TAX-%'`;
            } else if (saleType === 'product') {
                sql += ` AND (s.SaleType != 'service' OR s.SaleType IS NULL) AND (s.Barcode NOT LIKE 'S%' OR s.Barcode IS NULL)
                         AND s.Barcode NOT LIKE 'BED-%' AND s.Barcode NOT LIKE 'DISC-%' AND s.Barcode NOT LIKE 'TAX-%'`;
            }
        }
        
        // Date filtering
        if (startDate && endDate) {
            sql += ' AND s.Date BETWEEN ? AND ?';
            params.push(startDate, endDate);
        } else if (startDate) {
            sql += ' AND s.Date >= ?';
            params.push(startDate);
        } else if (endDate) {
            sql += ' AND s.Date <= ?';
            params.push(endDate);
        }
        
        // Exclude product items that have been fully returned (services are not excluded as they don't affect stock)
        sql += ` AND (
                    -- Include services (they don't have stock returns)
                    s.SaleType = 'service' OR s.Barcode LIKE 'S%'
                    OR
                    -- Include products that haven't been fully returned
                    NOT EXISTS (
                        SELECT 1
                        FROM tbl_sale_return_item sri
                        INNER JOIN tbl_voucher_return vr ON sri.vno = vr.id
                        WHERE vr.customer_name = CONCAT('Return for Voucher ', s.VNo)
                        AND sri.barcode = s.Barcode
                        GROUP BY sri.barcode
                        HAVING SUM(sri.qty) >= s.Qty
                    )
                )`;
        
        sql += ' ORDER BY s.Date DESC, s.VNo DESC, s.id ASC';
        
        console.log('Executing comprehensive sales query with params:', params);
        const results = await query(sql, params);
        console.log(`Found ${results ? results.length : 0} sales records`);
        
        res.json({ 
            success: true, 
            data: results || [],
            summary: {
                total_records: results ? results.length : 0,
                total_sales: results ? results.reduce((sum, item) => sum + parseFloat(item.total || 0), 0) : 0,
                total_profit: results ? results.reduce((sum, item) => sum + parseFloat(item.profit || 0), 0) : 0,
                unique_vouchers: results ? [...new Set(results.map(item => item.voucher_no))].length : 0
            }
        });
        
    } catch (error) {
        console.error('Comprehensive Sales Error:', error.message);
        console.error('Error details:', error);
        next(error);
    }
});

/**
 * @desc    Get Profit & Loss Report
 * @route   GET /api/reports/profit-loss
 * @access  Private
 */
router.get('/profit-loss', async (req, res, next) => {
    try {
        const { from, to } = req.query;
        
        if (!from || !to) {
            return res.status(400).json({
                success: false,
                message: 'From date and to date are required'
            });
        }

        const params = [from, to];

        // 1. Stock Sales Revenue (from tbl_sale where items are stock/pharmacy items)
        // Stock items: NOT services (SaleType != 'service' AND Barcode NOT LIKE 'S%') AND NOT lab items
        // Exclude Inpatient sales (those are calculated separately)
        const stockSalesQuery = `
            SELECT COALESCE(SUM(COALESCE(s.Profit, s.Total - COALESCE(s.cost_price, 0) * s.Qty)), 0) as revenue,
                   COALESCE(SUM(s.Total), 0) as total_sales
            FROM tbl_sale s
            WHERE DATE(s.Date) BETWEEN ? AND ?
            AND (s.SaleType IS NULL OR s.SaleType != 'Inpatient')
            AND (s.SaleType != 'service' OR s.SaleType IS NULL)
            AND (s.Barcode NOT LIKE 'S%' OR s.Barcode IS NULL)
            AND (s.Barcode NOT LIKE 'LAB-%' OR s.Barcode IS NULL)
            AND (s.Barcode NOT LIKE 'L%' OR s.Barcode IS NULL)
        `;
        const [stockSalesResult] = await query(stockSalesQuery, params);
        const stockSales = parseFloat(stockSalesResult?.revenue || 0);
        const stockSalesTotal = parseFloat(stockSalesResult?.total_sales || 0);

        // 2. Service Sales Revenue (from tbl_sale where items are services)
        // Services: SaleType = 'service' OR Barcode LIKE 'S%'
        // Exclude Inpatient sales
        const serviceSalesQuery = `
            SELECT COALESCE(SUM(s.Total), 0) as revenue
            FROM tbl_sale s
            WHERE DATE(s.Date) BETWEEN ? AND ?
            AND (s.SaleType IS NULL OR s.SaleType != 'Inpatient')
            AND (s.SaleType = 'service' OR s.Barcode LIKE 'S%')
        `;
        const [serviceSalesResult] = await query(serviceSalesQuery, params);
        const serviceSales = parseFloat(serviceSalesResult?.revenue || 0);

        // 3. Lab Items Sales Revenue (from tbl_sale where items are lab items)
        // Lab items: Barcode LIKE 'LAB-%' OR Barcode LIKE 'L%' (but not services starting with S)
        // Exclude Inpatient sales
        const labSalesQuery = `
            SELECT COALESCE(SUM(s.Total), 0) as revenue
            FROM tbl_sale s
            WHERE DATE(s.Date) BETWEEN ? AND ?
            AND (s.SaleType IS NULL OR s.SaleType != 'Inpatient')
            AND (s.Barcode LIKE 'LAB-%' OR (s.Barcode LIKE 'L%' AND s.Barcode NOT LIKE 'S%'))
        `;
        const [labSalesResult] = await query(labSalesQuery, params);
        const labSales = parseFloat(labSalesResult?.revenue || 0);

        // 4. Inpatient Sales Revenue
        const inpatientSalesQuery = `
            SELECT COALESCE(SUM(COALESCE(s.Profit, s.Total - COALESCE(s.cost_price, 0) * s.Qty)), 0) as revenue,
                   COALESCE(SUM(s.Total), 0) as total_sales
            FROM tbl_sale s
            WHERE DATE(s.Date) BETWEEN ? AND ?
            AND s.SaleType = 'Inpatient'
            AND s.Barcode NOT LIKE 'BED-%'
            AND s.Barcode NOT LIKE 'DISC-%'
            AND s.Barcode NOT LIKE 'TAX-%'
        `;
        const [inpatientSalesResult] = await query(inpatientSalesQuery, params);
        const inpatientSales = parseFloat(inpatientSalesResult?.revenue || 0);
        const inpatientSalesTotal = parseFloat(inpatientSalesResult?.total_sales || 0);

        // 5. Total Expenses (from expenses table if exists)
        let totalExpenses = 0;
        try {
            const expensesQuery = `
                SELECT COALESCE(SUM(amount), 0) as total
                FROM tbl_expenses
                WHERE DATE(date) BETWEEN ? AND ?
            `;
            const [expensesResult] = await query(expensesQuery, params);
            totalExpenses = parseFloat(expensesResult?.total || 0);
        } catch (err) {
            console.log('Expenses table may not exist:', err.message);
        }

        // 5b. Stock Expenses (from tbl_stock_expense)
        let stockExpenses = 0;
        try {
            const stockExpensesQuery = `
                SELECT COALESCE(SUM(total_cost), 0) as total
                FROM tbl_stock_expense
                WHERE DATE(date) BETWEEN ? AND ?
            `;
            const [stockExpensesResult] = await query(stockExpensesQuery, params);
            stockExpenses = parseFloat(stockExpensesResult?.total || 0);
        } catch (err) {
            console.log('Stock expenses calculation error:', err.message);
        }

        // 6. Expired Items Loss (from stock where items expired)
        let expiredItems = 0;
        try {
            const expiredQuery = `
                SELECT COALESCE(SUM(s.cost_price * s.qty), 0) as loss
                FROM tbl_stock s
                WHERE DATE(s.expire_date) BETWEEN ? AND ?
                AND s.expire_date < CURDATE()
                AND s.qty > 0
            `;
            const [expiredResult] = await query(expiredQuery, params);
            expiredItems = parseFloat(expiredResult?.loss || 0);
        } catch (err) {
            console.log('Expired items calculation error:', err.message);
        }

        // 7. Refer Fees (from external referring doctor commissions)
        let referFees = 0;
        try {
            const referFeesQuery = `
                SELECT COALESCE(SUM(rc.commission_amount), 0) as total
                FROM tbl_referral_commission rc
                INNER JOIN tbl_referring_doctor rd ON rc.referring_doctor_id = rd.id
                WHERE DATE(rc.transaction_date) BETWEEN ? AND ?
                AND rc.payment_status IN ('approved', 'paid')
                AND (rd.is_internal = 0 OR rd.is_internal IS NULL)
            `;
            const [referFeesResult] = await query(referFeesQuery, params);
            referFees = parseFloat(referFeesResult?.total || 0);
        } catch (err) {
            console.log('Refer fees calculation error:', err.message);
        }

        // 8. Commission Fees (from internal doctor commissions)
        let commissionFees = 0;
        try {
            const commissionFeesQuery = `
                SELECT COALESCE(SUM(rc.commission_amount), 0) as total
                FROM tbl_referral_commission rc
                INNER JOIN tbl_referring_doctor rd ON rc.referring_doctor_id = rd.id
                WHERE DATE(rc.transaction_date) BETWEEN ? AND ?
                AND rc.payment_status IN ('approved', 'paid')
                AND rd.is_internal = 1
            `;
            const [commissionFeesResult] = await query(commissionFeesQuery, params);
            commissionFees = parseFloat(commissionFeesResult?.total || 0);
        } catch (err) {
            console.log('Commission fees calculation error:', err.message);
        }

        // Calculate totals
        const totalRevenue = stockSalesTotal + serviceSales + labSales + inpatientSalesTotal;
        totalExpenses = totalExpenses + stockExpenses + expiredItems + referFees + commissionFees;
        const netProfit = totalRevenue - totalExpenses;

        res.json({
            success: true,
            data: {
                stockSales: Math.round(stockSales),
                serviceSales: Math.round(serviceSales),
                labSales: Math.round(labSales),
                inpatientSales: Math.round(inpatientSales),
                totalRevenue: Math.round(totalRevenue),
                totalExpenses: Math.round(totalExpenses),
                stockExpenses: Math.round(stockExpenses),
                expiredItems: Math.round(expiredItems),
                referFees: Math.round(referFees),
                commissionFees: Math.round(commissionFees),
                otherExpenses: Math.round(totalExpenses - stockExpenses - expiredItems - referFees - commissionFees),
                netProfit: Math.round(netProfit),
                period: {
                    from,
                    to
                }
            }
        });
    } catch (error) {
        console.error('Profit & Loss Report Error:', error);
        next(error);
    }
});

module.exports = router;

