const express = require('express');
const router = express.Router();
const { protect } = require('../middleware/auth');
const { query, transaction } = require('../config/database');

router.use(protect);

// Get all sale returns
router.get('/', async (req, res, next) => {
    try {
        const { startDate, endDate, voucherNo } = req.query;
        
        let sql = `
            SELECT 
                vr.id as return_id,
                vr.Date as return_date,
                vr.user_name as processed_by,
                vr.actual_cost as total_return_amount,
                vr.status,
                vr.customer_name as original_voucher_info,
                COUNT(sri.id) as items_count
            FROM tbl_voucher_return vr
            LEFT JOIN tbl_sale_return_item sri ON sri.vno = vr.id
            WHERE 1=1
        `;
        
        const params = [];
        
        if (startDate) {
            sql += ' AND DATE(vr.Date) >= ?';
            params.push(startDate);
        }
        
        if (endDate) {
            sql += ' AND DATE(vr.Date) <= ?';
            params.push(endDate);
        }
        
        
        sql += ' GROUP BY vr.id, vr.Date, vr.user_name, vr.actual_cost, vr.status, vr.customer_name ORDER BY vr.Date DESC LIMIT 100';
        
        const returns = await query(sql, params);
        res.json({ success: true, data: returns || [] });
        
    } catch (error) {
        console.error('❌ Error fetching returns:', error);
        console.error('Query params:', req.query);
        res.status(500).json({
            success: false,
            message: 'Error fetching returns: ' + error.message
        });
    }
});

// Get return details
router.get('/:returnId', async (req, res, next) => {
    try {
        const returnId = req.params.returnId;
        
        // Get return header
        const returnSql = `SELECT * FROM tbl_voucher_return WHERE id = ?`;
        const returnData = await query(returnSql, [returnId]);
        
        if (returnData.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Return not found'
            });
        }
        
        // Get return items
        const itemsSql = `SELECT * FROM tbl_sale_return_item WHERE vno = ?`;
        const items = await query(itemsSql, [returnId]);
        
        res.json({
            success: true,
            data: {
                return: returnData[0],
                items: items || []
            }
        });
        
    } catch (error) {
        console.error('Error fetching return details:', error);
        next(error);
    }
});

// Process sale return
router.post('/process', async (req, res, next) => {
    try {
        const {
            originalVoucherNo,
            items, // Array of {barcode, name, qty, price, total}
            reason,
            totalAmount
        } = req.body;
        
        console.log('🔄 Processing sale return:', { originalVoucherNo, itemsCount: items.length, totalAmount });
        
        // Validate input
        if (!originalVoucherNo || !items || items.length === 0) {
            return res.status(400).json({
                success: false,
                message: 'Original voucher number and items are required'
            });
        }
        
        // Check what has already been returned for this voucher
        const checkReturnedSql = `
            SELECT sri.barcode, SUM(sri.qty) as returned_qty
            FROM tbl_sale_return_item sri
            INNER JOIN tbl_voucher_return vr ON sri.vno = vr.id
            WHERE vr.customer_name LIKE ?
            GROUP BY sri.barcode
        `;
        const alreadyReturned = await query(checkReturnedSql, [`Return for Voucher ${originalVoucherNo}`]);
        const returnedMap = {};
        alreadyReturned.forEach(item => {
            returnedMap[item.barcode] = parseInt(item.returned_qty);
        });
        
        // Get original sale items to validate returnable quantities and get warehouse_id
        const originalItemsSql = 'SELECT Barcode, Qty, warehouse_id, id as sale_id FROM tbl_sale WHERE VNo = ?';
        const originalItems = await query(originalItemsSql, [originalVoucherNo]);
        const originalMap = {};
        const originalWarehouseMap = {}; // Map barcode to warehouse_id
        originalItems.forEach(item => {
            originalMap[item.Barcode] = parseInt(item.Qty);
            // Store warehouse_id for each barcode (use first non-null warehouse_id found)
            if (item.warehouse_id && !originalWarehouseMap[item.Barcode]) {
                originalWarehouseMap[item.Barcode] = item.warehouse_id;
            }
        });
        
        // Validate return quantities
        for (const returnItem of items) {
            const originalQty = originalMap[returnItem.barcode] || 0;
            const alreadyReturnedQty = returnedMap[returnItem.barcode] || 0;
            const returnableQty = originalQty - alreadyReturnedQty;
            
            if (returnItem.qty > returnableQty) {
                return res.status(400).json({
                    success: false,
                    message: `Cannot return ${returnItem.qty} units of ${returnItem.name}. Only ${returnableQty} units are returnable (${originalQty} sold, ${alreadyReturnedQty} already returned).`
                });
            }
        }
        
        // Use transaction to ensure data consistency
        const result = await transaction(async (conn) => {
            // Create voucher return record (matching existing schema)
            const [voucherReturnResult] = await conn.query(
                `INSERT INTO tbl_voucher_return 
                (user_name, sub_total, actual_cost, dis, Date, status, customer_name)
                VALUES (?, ?, ?, ?, NOW(), ?, ?)`,
                [
                    req.user.name || req.user.username,
                    totalAmount,
                    totalAmount,
                    0,
                    1, // status: 1 = completed
                    `Return for Voucher ${originalVoucherNo}`
                ]
            );
            
            const returnId = voucherReturnResult.insertId;
            console.log('🔄 Created return voucher:', returnId);
            
            // Get original voucher to check if it exists and get current amounts
            const [voucherData] = await conn.query(
                'SELECT * FROM tbl_voucher WHERE id = ?',
                [originalVoucherNo]
            );
            
            if (voucherData.length === 0) {
                throw new Error(`Original voucher ${originalVoucherNo} not found`);
            }
            
            const originalVoucher = voucherData[0];
            
            // Process each return item
            for (const item of items) {
                // Insert return item
                await conn.query(
                    `INSERT INTO tbl_sale_return_item 
                    (cat_id, barcode, name, sale_price, qty, total, remark, date, vno)
                    VALUES (?, ?, ?, ?, ?, ?, ?, CURDATE(), ?)`,
                    [
                        item.cat_id || null,
                        item.barcode,
                        item.name,
                        item.price,
                        item.qty,
                        item.total,
                        reason || 'Customer return',
                        returnId
                    ]
                );
                
                // Check if it's a product (not a service)
                const isService = item.barcode && item.barcode.startsWith('S');
                
                if (!isService) {
                    // Get stock information
                    const [stockCheck] = await conn.query(
                        'SELECT * FROM tbl_stock WHERE barcode = ?',
                        [item.barcode]
                    );
                    
                    if (stockCheck.length > 0) {
                        const stockId = stockCheck[0].id;
                        const warehouseId = originalWarehouseMap[item.barcode] || null;
                        
                        // Always update main stock for backward compatibility
                        await conn.query(
                            'UPDATE tbl_stock SET qty = qty + ? WHERE barcode = ?',
                            [item.qty, item.barcode]
                        );
                        
                        // If warehouse_id exists, return stock to that warehouse
                        if (warehouseId) {
                            try {
                                // Create stock movement record to return stock to warehouse
                                // The trigger will automatically update warehouse stock
                                await conn.query(
                                    `INSERT INTO tbl_stock_movement
                                     (movement_type, warehouse_id, stock_id, quantity, reference_type, reference_id,
                                      movement_date, notes, created_by)
                                     VALUES ('IN', ?, ?, ?, 'RETURN', ?, NOW(), ?, ?)`,
                                    [
                                        warehouseId,
                                        stockId,
                                        item.qty,
                                        returnId,
                                        `Sale Return: ${item.name} (Voucher ${originalVoucherNo})`,
                                        req.user?.id || null
                                    ]
                                );
                                console.log(`✅ Returned ${item.qty} units of ${item.name} to warehouse ${warehouseId}`);
                            } catch (warehouseError) {
                                console.error('Error returning stock to warehouse:', warehouseError.message);
                                // Continue even if warehouse update fails - main stock was already updated
                            }
                        } else {
                            console.log(`✅ Returned ${item.qty} units of ${item.name} to main stock (no warehouse specified)`);
                        }
                        
                        // Remove low stock alert if stock is now above threshold
                        const newQty = stockCheck[0].qty + item.qty;
                        if (newQty > stockCheck[0].qty_alert) {
                            await conn.query(
                                'DELETE FROM tbl_alert WHERE barcode = ?',
                                [item.barcode]
                            );
                        }
                    }
                } else {
                    console.log(`ℹ️ Service item ${item.name} - no stock update needed`);
                }
            }
            
            // Update original voucher amounts by subtracting the return amount
            const newSubTotal = Math.max(0, (parseFloat(originalVoucher.sub_total) || 0) - totalAmount);
            const newActualCost = Math.max(0, (parseFloat(originalVoucher.actual_cost) || 0) - totalAmount);
            
            await conn.query(
                `UPDATE tbl_voucher 
                SET sub_total = ?, actual_cost = ? 
                WHERE id = ?`,
                [newSubTotal, newActualCost, originalVoucherNo]
            );
            
            console.log(`✅ Updated voucher ${originalVoucherNo} amounts: sub_total ${originalVoucher.sub_total} -> ${newSubTotal}, actual_cost ${originalVoucher.actual_cost} -> ${newActualCost}`);
            
            return {
                returnId,
                originalVoucherNo,
                itemsReturned: items.length,
                totalAmount,
                voucherUpdated: true,
                newVoucherSubTotal: newSubTotal,
                newVoucherActualCost: newActualCost
            };
        });
        
        console.log('✅ Return processed successfully:', result);
        
        res.status(201).json({
            success: true,
            message: 'Return processed successfully',
            data: result
        });
        
    } catch (error) {
        console.error('❌ Error processing return:', error);
        next(error);
    }
});

// Get original voucher for return
router.get('/voucher/:voucherNo', async (req, res, next) => {
    try {
        const voucherNo = req.params.voucherNo;
        
        console.log('🔍 Fetching original voucher for return:', voucherNo);
        
        // Get original voucher
        const voucherSql = 'SELECT * FROM tbl_voucher WHERE id = ?';
        const voucher = await query(voucherSql, [voucherNo]);
        
        if (voucher.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Voucher not found'
            });
        }
        
        // Get original sale items
        const itemsSql = 'SELECT * FROM tbl_sale WHERE VNo = ?';
        const items = await query(itemsSql, [voucherNo]);
        
        // Check if any items have already been returned for this voucher
        // We need to find return vouchers that reference this original voucher
        // The customer_name field stores "Return for Voucher #X" format
        const returnedSql = `
            SELECT sri.barcode, SUM(sri.qty) as returned_qty
            FROM tbl_sale_return_item sri
            INNER JOIN tbl_voucher_return vr ON sri.vno = vr.id
            WHERE vr.customer_name LIKE ?
            GROUP BY sri.barcode
        `;
        const returnedItems = await query(returnedSql, [`Return for Voucher ${voucherNo}`]);
        
        // Create a map of returned quantities
        const returnedMap = {};
        returnedItems.forEach(item => {
            returnedMap[item.barcode] = parseInt(item.returned_qty);
        });
        
        // Add returnable quantity to each item
        const itemsWithReturnableQty = items.map(item => ({
            ...item,
            returned_qty: returnedMap[item.Barcode] || 0,
            returnable_qty: item.Qty - (returnedMap[item.Barcode] || 0)
        }));
        
        res.json({
            success: true,
            data: {
                voucher: voucher[0],
                items: itemsWithReturnableQty
            }
        });
        
    } catch (error) {
        console.error('❌ Error fetching voucher for return:', error);
        next(error);
    }
});

module.exports = router;

