const express = require('express');
const router = express.Router();
const { protect } = require('../middleware/auth');
const { requireCrudPermission } = require('../middleware/permissions');
const { query } = require('../config/database');

router.use(protect);
router.use(requireCrudPermission('sales'));

// Get all sales
router.get('/', async (req, res, next) => {
    try {
        const { date, vno, start_date, end_date, limit } = req.query;
        let sql;
        let params = [];
        
        // If start_date and end_date are provided, group by voucher (for dashboard)
        if (start_date && end_date) {
            sql = `SELECT 
                    s.VNo, 
                    MAX(s.Date) as Date, 
                    MAX(s.Cashier) as Cashier,
                    COALESCE(SUM(s.Total), 0) as total,
                    COALESCE(MAX(v.sub_total), COALESCE(SUM(s.Total), 0) + COALESCE(MAX(v.dis), 0)) as subtotal,
                    COALESCE(MAX(v.dis), 0) as discount,
                    COALESCE(MAX(v.actual_cost), COALESCE(SUM(s.Total), 0)) as actual_total,
                    COALESCE(SUM(s.Profit), 0) as profit,
                    COUNT(s.id) as items_count,
                    COALESCE(p.name, COALESCE(MAX(v.customer_name), 'Walk-in')) as customer_name
                   FROM tbl_sale s
                   LEFT JOIN tbl_voucher v ON s.VNo = v.id
                   LEFT JOIN tbl_patient p ON v.patient_id = p.id
                   WHERE DATE(s.Date) BETWEEN ? AND ?
                     AND s.Total IS NOT NULL
                     AND s.Total > 0`;
            params.push(start_date, end_date);
            sql += ' GROUP BY s.VNo';
            sql += ' HAVING total > 0';
            sql += ' ORDER BY MAX(s.Date) DESC, s.VNo DESC';
            if (limit) {
                sql += ` LIMIT ${parseInt(limit) || 10}`;
            } else {
                sql += ' LIMIT 100';
            }
        } else if (date) {
            // Single date - group by voucher
            // Join with tbl_voucher to get discount, subtotal, and calculate profit
            sql = `SELECT 
                    s.VNo, 
                    MAX(s.Date) as Date, 
                    MAX(s.Cashier) as Cashier,
                    COALESCE(SUM(s.Total), 0) as total,
                    COALESCE(MAX(v.sub_total), COALESCE(SUM(s.Total), 0) + COALESCE(MAX(v.dis), 0)) as subtotal,
                    COALESCE(MAX(v.dis), 0) as discount,
                    COALESCE(MAX(v.actual_cost), COALESCE(SUM(s.Total), 0)) as actual_total,
                    COALESCE(SUM(s.Profit), 0) as profit,
                    COUNT(s.id) as items_count,
                    COALESCE(p.name, COALESCE(MAX(v.customer_name), 'Walk-in')) as customer_name
                   FROM tbl_sale s
                   LEFT JOIN tbl_voucher v ON s.VNo = v.id
                   LEFT JOIN tbl_patient p ON v.patient_id = p.id
                   WHERE s.Date = ?
                     AND s.Total IS NOT NULL
                     AND s.Total > 0`;
            params.push(date);
            sql += ' GROUP BY s.VNo';
            sql += ' HAVING total > 0';
            sql += ' ORDER BY MAX(s.Date) DESC, s.VNo DESC LIMIT 100';
        } else if (vno) {
            // Get specific voucher details
            sql = 'SELECT * FROM tbl_sale WHERE VNo = ?';
            params.push(vno);
            sql += ' ORDER BY id ASC';
        } else {
            // Default: get all sales grouped by voucher
            // Join with tbl_voucher to get discount, subtotal, and calculate profit
            sql = `SELECT 
                    s.VNo, 
                    MAX(s.Date) as Date, 
                    MAX(s.Cashier) as Cashier,
                    COALESCE(SUM(s.Total), 0) as total,
                    COALESCE(MAX(v.sub_total), COALESCE(SUM(s.Total), 0) + COALESCE(MAX(v.dis), 0)) as subtotal,
                    COALESCE(MAX(v.dis), 0) as discount,
                    COALESCE(MAX(v.actual_cost), COALESCE(SUM(s.Total), 0)) as actual_total,
                    COALESCE(SUM(s.Profit), 0) as profit,
                    COUNT(s.id) as items_count,
                    COALESCE(p.name, COALESCE(MAX(v.customer_name), 'Walk-in')) as customer_name
                   FROM tbl_sale s
                   LEFT JOIN tbl_voucher v ON s.VNo = v.id
                   LEFT JOIN tbl_patient p ON v.patient_id = p.id
                   WHERE s.Total IS NOT NULL AND s.Total > 0
                   GROUP BY s.VNo
                   HAVING total > 0
                   ORDER BY MAX(s.Date) DESC, s.VNo DESC LIMIT 100`;
        }
        
        const sales = await query(sql, params);
        res.json({ success: true, data: sales });
    } catch (error) {
        next(error);
    }
});

// Get voucher details
router.get('/voucher/:vno', async (req, res, next) => {
    try {
        const salesSql = 'SELECT * FROM tbl_sale WHERE VNo = ?';
        const voucherSql = `
            SELECT 
                v.*,
                p.name as patient_name,
                p.phone as patient_phone,
                p.address as patient_address
            FROM tbl_voucher v
            LEFT JOIN tbl_patient p ON v.patient_id = p.id
            WHERE v.id = ?
        `;
        
        const sales = await query(salesSql, [req.params.vno]);
        const voucherRows = await query(voucherSql, [req.params.vno]);
        let voucher = voucherRows[0] || null;

        if (!voucher && sales.length > 0) {
            const totals = sales.reduce((acc, item) => {
                const qty = parseFloat(item.Qty) || 0;
                const price = parseFloat(item.SalePrice) || 0;
                const discount = parseFloat(item.dis) || 0;
                const lineTotal = parseFloat(item.Total);
                acc.subTotal += qty * price;
                acc.discount += discount;
                acc.total += Number.isFinite(lineTotal) ? lineTotal : (qty * price - discount);
                acc.latestDate = acc.latestDate && item.Date
                    ? (new Date(item.Date) > new Date(acc.latestDate) ? item.Date : acc.latestDate)
                    : (item.Date || acc.latestDate);
                acc.cashier = item.Cashier || acc.cashier;
                return acc;
            }, {
                subTotal: 0,
                discount: 0,
                total: 0,
                latestDate: null,
                cashier: null
            });

            voucher = {
                id: req.params.vno,
                VNo: req.params.vno,
                Date: totals.latestDate || null,
                Time: null,
                doctor_name: null,
                patient_name: null,
                customer_name: totals.cashier || 'Walk-in',
                sub_total: totals.subTotal,
                actual_cost: totals.total,
                dis: totals.discount,
                payment: 'Cash',
                payment_method: 'Cash',
                Balance: totals.total
            };
        }

        if (voucher) {
            voucher.payment_method = voucher.payment_method || voucher.payment || 'Cash';
        }
        
        res.json({ success: true, data: { sales, voucher } });
    } catch (error) {
        next(error);
    }
});

// Get customer purchase history
router.get('/customer/:customerId', async (req, res, next) => {
    try {
        const customerId = req.params.customerId;
        
        // Get vouchers for this customer
        const voucherSql = `
            SELECT 
                v.id as voucher_id,
                v.VNo,
                v.Date,
                v.Time,
                v.Total,
                v.Discount,
                v.Balance,
                v.customer_id,
                v.created_at,
                COUNT(s.id) as items_count
            FROM tbl_voucher v
            LEFT JOIN tbl_sale s ON v.id = s.VNo
            WHERE v.customer_id = ?
            GROUP BY v.id
            ORDER BY v.Date DESC, v.Time DESC
            LIMIT 100
        `;
        
        const vouchers = await query(voucherSql, [customerId]);
        
        // Get detailed sales items for each voucher
        for (let voucher of vouchers) {
            const salesSql = `
                SELECT 
                    s.id,
                    s.product_id,
                    s.Item,
                    s.Qty,
                    s.Price,
                    s.Amount,
                    s.Discount as item_discount
                FROM tbl_sale s
                WHERE s.VNo = ?
                ORDER BY s.id
            `;
            voucher.items = await query(salesSql, [voucher.voucher_id]);
        }
        
        // Calculate statistics
        const stats = {
            total_purchases: vouchers.length,
            total_amount: vouchers.reduce((sum, v) => sum + (parseFloat(v.Total) || 0), 0),
            total_discount: vouchers.reduce((sum, v) => sum + (parseFloat(v.Discount) || 0), 0),
            total_items: vouchers.reduce((sum, v) => sum + (parseInt(v.items_count) || 0), 0)
        };
        
        res.json({ 
            success: true, 
            data: {
                vouchers,
                stats
            }
        });
    } catch (error) {
        console.error('Error fetching customer purchase history:', error);
        // Return empty data on error to avoid frontend issues
        res.json({ 
            success: true, 
            data: {
                vouchers: [],
                stats: {
                    total_purchases: 0,
                    total_amount: 0,
                    total_discount: 0,
                    total_items: 0
                }
            }
        });
    }
});

module.exports = router;

