/**
 * Service Routes
 */

const express = require('express');
const router = express.Router();
const { protect } = require('../middleware/auth');
const { query } = require('../config/database');

router.use(protect);

/**
 * @desc    Get all services
 * @route   GET /api/services
 * @access  Private
 */
router.get('/', async (req, res, next) => {
    try {
        const { page = 1, limit = 200, category, search } = req.query;
        const offset = (page - 1) * limit;

        let sql = `SELECT s.*, sc.name as category_name 
                   FROM tbl_service s 
                   LEFT JOIN tbl_service_catalog sc ON s.cat_id = sc.id
                   WHERE 1=1`;
        
        const params = [];

        if (category) {
            sql += ' AND s.cat_id = ?';
            params.push(category);
        }

        if (search) {
            sql += ' AND (s.service_name LIKE ? OR s.barcode LIKE ?)';
            params.push(`%${search}%`, `%${search}%`);
        }

        sql += ' ORDER BY s.service_name LIMIT ? OFFSET ?';
        params.push(parseInt(limit), parseInt(offset));

        const services = await query(sql, params);

        // Get total count
        let countSql = `SELECT COUNT(*) as total FROM tbl_service s WHERE 1=1`;
        const countParams = [];
        
        if (category) {
            countSql += ' AND s.cat_id = ?';
            countParams.push(category);
        }
        
        if (search) {
            countSql += ' AND (s.service_name LIKE ? OR s.barcode LIKE ?)';
            countParams.push(`%${search}%`, `%${search}%`);
        }

        const countResult = await query(countSql, countParams);

        res.json({
            success: true,
            count: services.length,
            total: countResult[0].total,
            page: parseInt(page),
            pages: Math.ceil(countResult[0].total / limit),
            data: services
        });
    } catch (error) {
        console.error('Error fetching services:', error);
        res.json({
            success: true,
            data: [],
            total: 0,
            count: 0,
            message: 'No services found'
        });
    }
});

/**
 * @desc    Get service categories
 * @route   GET /api/services/categories/list
 * @access  Private
 */
router.get('/categories/list', async (req, res, next) => {
    try {
        const sql = 'SELECT * FROM tbl_service_catalog ORDER BY name';
        const categories = await query(sql);
        
        res.json({
            success: true,
            data: categories
        });
    } catch (error) {
        console.error('Error fetching service categories:', error);
        res.json({
            success: true,
            data: []
        });
    }
});

/**
 * @desc    Get all service categories (for CRUD)
 * @route   GET /api/services/categories
 * @access  Private
 */
router.get('/categories', async (req, res, next) => {
    try {
        const sql = `SELECT sc.*, 
                            COUNT(s.id) as service_count
                     FROM tbl_service_catalog sc
                     LEFT JOIN tbl_service s ON sc.id = s.cat_id
                     GROUP BY sc.id, sc.name
                     ORDER BY sc.name`;
        const categories = await query(sql);
        
        // Convert service_count to number
        const categoriesWithCount = categories.map(cat => ({
            ...cat,
            service_count: parseInt(cat.service_count) || 0,
            is_used: (parseInt(cat.service_count) || 0) > 0
        }));
        
        res.json({
            success: true,
            data: categoriesWithCount
        });
    } catch (error) {
        console.error('Error fetching service categories:', error);
        next(error);
    }
});

/**
 * @desc    Add service category
 * @route   POST /api/services/categories
 * @access  Private
 */
router.post('/categories', async (req, res, next) => {
    try {
        const { name } = req.body;

        if (!name || !name.trim()) {
            return res.status(400).json({
                success: false,
                message: 'Category name is required'
            });
        }

        // Check if category already exists
        const existing = await query('SELECT id FROM tbl_service_catalog WHERE name = ?', [name.trim()]);
        if (existing.length > 0) {
            return res.status(400).json({
                success: false,
                message: 'Category with this name already exists'
            });
        }

        const sql = 'INSERT INTO tbl_service_catalog (name) VALUES (?)';
        const result = await query(sql, [name.trim()]);

        res.status(201).json({
            success: true,
            message: 'Service category added successfully',
            data: { id: result.insertId, name: name.trim() }
        });
    } catch (error) {
        console.error('Error adding service category:', error);
        next(error);
    }
});

/**
 * @desc    Update service category
 * @route   PUT /api/services/categories/:id
 * @access  Private
 */
router.put('/categories/:id', async (req, res, next) => {
    try {
        const { name } = req.body;
        const { id } = req.params;

        if (!name || !name.trim()) {
            return res.status(400).json({
                success: false,
                message: 'Category name is required'
            });
        }

        // Check if category exists
        const existing = await query('SELECT id FROM tbl_service_catalog WHERE id = ?', [id]);
        if (existing.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Service category not found'
            });
        }

        // Check if another category with same name exists
        const duplicate = await query('SELECT id FROM tbl_service_catalog WHERE name = ? AND id != ?', [name.trim(), id]);
        if (duplicate.length > 0) {
            return res.status(400).json({
                success: false,
                message: 'Category with this name already exists'
            });
        }

        const sql = 'UPDATE tbl_service_catalog SET name = ? WHERE id = ?';
        await query(sql, [name.trim(), id]);

        res.json({
            success: true,
            message: 'Service category updated successfully',
            data: { id: parseInt(id), name: name.trim() }
        });
    } catch (error) {
        console.error('Error updating service category:', error);
        next(error);
    }
});

/**
 * @desc    Delete service category
 * @route   DELETE /api/services/categories/:id
 * @access  Private
 */
router.delete('/categories/:id', async (req, res, next) => {
    try {
        const { id } = req.params;

        // Check if category exists
        const existing = await query('SELECT id FROM tbl_service_catalog WHERE id = ?', [id]);
        if (existing.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Service category not found'
            });
        }

        // Check if category is being used by any services
        const servicesUsing = await query('SELECT COUNT(*) as count FROM tbl_service WHERE cat_id = ?', [id]);
        if (servicesUsing[0].count > 0) {
            return res.status(400).json({
                success: false,
                message: `Cannot delete category. It is being used by ${servicesUsing[0].count} service(s)`
            });
        }

        const sql = 'DELETE FROM tbl_service_catalog WHERE id = ?';
        await query(sql, [id]);

        res.json({
            success: true,
            message: 'Service category deleted successfully'
        });
    } catch (error) {
        console.error('Error deleting service category:', error);
        next(error);
    }
});

/**
 * @desc    Get single service
 * @route   GET /api/services/:id
 * @access  Private
 */
router.get('/:id', async (req, res, next) => {
    try {
        const sql = `SELECT s.*, sc.name as category_name 
                     FROM tbl_service s 
                     LEFT JOIN tbl_service_catalog sc ON s.cat_id = sc.id
                     WHERE s.id = ?`;
        
        const services = await query(sql, [req.params.id]);

        if (services.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Service not found'
            });
        }

        res.json({
            success: true,
            data: services[0]
        });
    } catch (error) {
        console.error('Error fetching service:', error);
        res.status(500).json({
            success: false,
            message: 'Error fetching service details'
        });
    }
});

/**
 * @desc    Create new service
 * @route   POST /api/services
 * @access  Private
 */
router.post('/', async (req, res, next) => {
    try {
        const {
            barcode,
            service_name,
            cat_id,
            buy_price,
            sale_price,
            remark
        } = req.body;

        // Validate required fields
        if (!service_name) {
            return res.status(400).json({
                success: false,
                message: 'Service name is required'
            });
        }

        // Check if barcode already exists
        if (barcode) {
            const existingService = await query('SELECT id FROM tbl_service WHERE barcode = ?', [barcode]);
            if (existingService.length > 0) {
                return res.status(400).json({
                    success: false,
                    message: 'Service with this barcode already exists'
                });
            }
        }

        const sql = `INSERT INTO tbl_service 
                     (barcode, service_name, cat_id, buy_price, sale_price, remark)
                     VALUES (?, ?, ?, ?, ?, ?)`;

        const result = await query(sql, [
            barcode || null,
            service_name,
            cat_id || null,
            parseFloat(buy_price) || 0,
            parseFloat(sale_price) || 0,
            remark || null
        ]);

        res.status(201).json({
            success: true,
            message: 'Service created successfully',
            data: {
                id: result.insertId,
                barcode,
                service_name,
                cat_id,
                buy_price,
                sale_price
            }
        });
    } catch (error) {
        console.error('Error creating service:', error);
        res.status(500).json({
            success: false,
            message: 'Error creating service: ' + error.message
        });
    }
});

/**
 * @desc    Update service
 * @route   PUT /api/services/:id
 * @access  Private
 */
router.put('/:id', async (req, res, next) => {
    try {
        const {
            barcode,
            service_name,
            cat_id,
            buy_price,
            sale_price,
            remark
        } = req.body;

        // Validate required fields
        if (!service_name) {
            return res.status(400).json({
                success: false,
                message: 'Service name is required'
            });
        }

        // Get existing service
        const existingServices = await query('SELECT * FROM tbl_service WHERE id = ?', [req.params.id]);
        
        if (existingServices.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Service not found'
            });
        }

        // Check if barcode already exists (excluding current service)
        if (barcode) {
            const existingService = await query('SELECT id FROM tbl_service WHERE barcode = ? AND id != ?', [barcode, req.params.id]);
            if (existingService.length > 0) {
                return res.status(400).json({
                    success: false,
                    message: 'Service with this barcode already exists'
                });
            }
        }

        const sql = `UPDATE tbl_service 
                     SET barcode = ?, service_name = ?, cat_id = ?, 
                         buy_price = ?, sale_price = ?, remark = ?
                     WHERE id = ?`;

        const result = await query(sql, [
            barcode || null,
            service_name,
            cat_id || null,
            parseFloat(buy_price) || 0,
            parseFloat(sale_price) || 0,
            remark || null,
            req.params.id
        ]);

        res.json({
            success: true,
            message: 'Service updated successfully',
            data: {
                id: req.params.id,
                affectedRows: result.affectedRows
            }
        });
    } catch (error) {
        console.error('Error updating service:', error);
        res.status(500).json({
            success: false,
            message: 'Error updating service: ' + error.message
        });
    }
});

/**
 * @desc    Delete service
 * @route   DELETE /api/services/:id
 * @access  Private
 */
router.delete('/:id', async (req, res, next) => {
    try {
        const services = await query('SELECT service_name FROM tbl_service WHERE id = ?', [req.params.id]);
        
        if (services.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Service not found'
            });
        }

        await query('DELETE FROM tbl_service WHERE id = ?', [req.params.id]);

        res.json({
            success: true,
            message: 'Service deleted successfully'
        });
    } catch (error) {
        console.error('Error deleting service:', error);
        res.status(500).json({
            success: false,
            message: 'Error deleting service: ' + error.message
        });
    }
});

/**
 * @desc    Generate barcode for service
 * @route   GET /api/services/generate-barcode
 * @access  Private
 */
router.get('/generate-barcode/new', async (req, res, next) => {
    try {
        // Fetch recent barcodes and auto-increment the highest numeric value (strip non-digits)
        const rows = await query(
            `SELECT barcode FROM tbl_service 
             WHERE barcode IS NOT NULL AND barcode <> '' 
             ORDER BY id DESC 
             LIMIT 5000`
        );

        const toDigits = (val) => String(val || '').replace(/\D/g, '');

        const numericEntries = rows
            .map(r => {
                const digits = toDigits(r.barcode);
                return digits.length ? { digits, len: digits.length, num: parseInt(digits, 10) } : null;
            })
            .filter(e => e && !isNaN(e.num));

        let barcode = '100001'; // default starting point when no barcodes exist

        if (numericEntries.length) {
            const maxEntry = numericEntries.reduce((max, entry) => {
                if (entry.num > max.num) return entry;
                return max;
            }, { num: -1, len: 6, digits: '' });

            const nextNumber = maxEntry.num + 1;
            const targetLength = Math.max(maxEntry.len, String(nextNumber).length);
            barcode = String(nextNumber).padStart(targetLength, '0');
        }

        res.json({ success: true, barcode });
    } catch (error) {
        console.error('Error generating barcode:', error);
        res.status(500).json({
            success: false,
            message: 'Error generating barcode'
        });
    }
});

module.exports = router;
