/**
 * Settings Routes
 */

const express = require('express');
const router = express.Router();
const { protect } = require('../middleware/auth');
const { requireCrudPermission } = require('../middleware/permissions');
const multer = require('multer');
const path = require('path');
const fs = require('fs');

const {
    getShopInfo,
    updateShopInfo,
    getConfig,
    backupDatabase,
    getCategories,
    addCategory,
    updateCategory,
    deleteCategory,
    checkLogoPath,
    listLogos
} = require('../controllers/settings.controller');

// Create uploads/logo directory if it doesn't exist
const logoUploadDir = path.join(__dirname, '../uploads/logo');
if (!fs.existsSync(logoUploadDir)) {
    fs.mkdirSync(logoUploadDir, { recursive: true });
}

// Logo upload configuration
const logoStorage = multer.diskStorage({
    destination: (req, file, cb) => {
        cb(null, logoUploadDir);
    },
    filename: (req, file, cb) => {
        const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
        const ext = path.extname(file.originalname);
        cb(null, `logo-${uniqueSuffix}${ext}`);
    }
});

const logoUpload = multer({
    storage: logoStorage,
    limits: {
        fileSize: 5 * 1024 * 1024 // 5MB
    },
    fileFilter: (req, file, cb) => {
        const allowedTypes = /jpeg|jpg|png|gif|webp/;
        const extname = allowedTypes.test(path.extname(file.originalname).toLowerCase());
        const mimetype = allowedTypes.test(file.mimetype);

        if (mimetype && extname) {
            return cb(null, true);
        } else {
            cb(new Error('Only image files are allowed'));
        }
    }
}).single('logo');

// Protect all routes
router.use(protect);
router.use(requireCrudPermission('settings'));

// Shop info routes
router.get('/shop', getShopInfo);
router.put('/shop', (req, res, next) => {
    logoUpload(req, res, (err) => {
        if (err) {
            return res.status(400).json({
                success: false,
                message: err.message
            });
        }
        next();
    });
}, updateShopInfo);

// Configuration routes
router.get('/config', getConfig);
router.post('/backup', backupDatabase);

// Category management routes
router.get('/categories', getCategories);
router.post('/categories', addCategory);
router.put('/categories/:id', updateCategory);
router.delete('/categories/:id', deleteCategory);

// Logo management routes
router.get('/logo-check', checkLogoPath);
router.get('/logo-list', listLogos);

module.exports = router;
