const express = require('express');
const router = express.Router();
const { protect } = require('../middleware/auth');
const { requireCrudPermission } = require('../middleware/permissions');
const { query } = require('../config/database');
const stockUtils = require('../utils/stock-utils');

router.use(protect);
router.use(requireCrudPermission('stock'));

// Get all stock
router.get('/', async (req, res, next) => {
    try {
        const { search, cat_id, include_expiry_status } = req.query;
        let sql = `SELECT 
                    s.*,
                    c.name as category_name,
                    sup.name as supplier_name,
                    sup.id as supplier_id,
                    DATEDIFF(s.expire_date, CURDATE()) as days_until_expiry,
                    CASE 
                        WHEN s.expire_date IS NULL THEN 'no_expiry'
                        WHEN s.expire_date < CURDATE() THEN 'expired'
                        WHEN s.alert_date IS NOT NULL AND s.alert_date <= CURDATE() THEN 'expiring_soon'
                        ELSE 'valid'
                    END as expiry_status
                   FROM tbl_stock s
                   LEFT JOIN tbl_catalog c ON s.cat_id = c.id
                   LEFT JOIN tbl_supplier sup ON s.supplier_id = sup.id`;
        let params = [];
        
        if (search) {
            sql += ' WHERE (s.name LIKE ? OR s.barcode LIKE ?)';
            params = [`%${search}%`, `%${search}%`];
        }
        
        sql += ' ORDER BY s.name';
        const stock = await query(sql, params);
        
        // Add expiry status details if requested
        if (include_expiry_status === 'true') {
            const stockWithStatus = stock.map(item => {
                const expiryStatus = stockUtils.checkExpiryStatus(item.expire_date, item.alert_date);
                return {
                    ...item,
                    ...expiryStatus
                };
            });
            return res.json({ success: true, count: stockWithStatus.length, data: stockWithStatus });
        }
        
        res.json({ success: true, count: stock.length, data: stock });
    } catch (error) {
        next(error);
    }
});

// Get stock alerts (low quantity)
router.get('/alerts', async (req, res, next) => {
    try {
        const sql = 'SELECT * FROM tbl_stock WHERE qty <= qty_alert AND qty > 0 ORDER BY qty';
        const alerts = await query(sql);
        res.json({ success: true, count: alerts.length, data: alerts });
    } catch (error) {
        next(error);
    }
});

// Get expiry alerts (must be before /:id route)
router.get('/expiry-alerts', async (req, res, next) => {
    try {
        const daysBeforeAlert = parseInt(req.query.days) || 30;
        const alerts = await stockUtils.getExpiryAlerts(daysBeforeAlert);
        res.json({ 
            success: true, 
            count: alerts.length, 
            data: alerts,
            summary: {
                expired: alerts.filter(a => a.isExpired).length,
                expiring_soon: alerts.filter(a => a.isExpiringSoon && !a.isExpired).length,
                total: alerts.length
            }
        });
    } catch (error) {
        next(error);
    }
});

// Get single stock item
router.get('/:id', async (req, res, next) => {
    try {
        const sql = `SELECT 
                    s.*,
                    c.name as category_name,
                    sup.name as supplier_name,
                    sup.id as supplier_id,
                    DATEDIFF(s.expire_date, CURDATE()) as days_until_expiry,
                    CASE 
                        WHEN s.expire_date IS NULL THEN 'no_expiry'
                        WHEN s.expire_date < CURDATE() THEN 'expired'
                        WHEN s.alert_date IS NOT NULL AND s.alert_date <= CURDATE() THEN 'expiring_soon'
                        ELSE 'valid'
                    END as expiry_status
                   FROM tbl_stock s
                   LEFT JOIN tbl_catalog c ON s.cat_id = c.id
                   LEFT JOIN tbl_supplier sup ON s.supplier_id = sup.id
                   WHERE s.id = ?`;
        
        const stock = await query(sql, [req.params.id]);
        
        if (stock.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Stock item not found'
            });
        }
        
        const item = stock[0];
        const expiryStatus = stockUtils.checkExpiryStatus(item.expire_date, item.alert_date);
        
        res.json({
            success: true,
            data: {
                ...item,
                ...expiryStatus
            }
        });
    } catch (error) {
        next(error);
    }
});

// Update stock price
router.put('/:id/price', async (req, res, next) => {
    try {
        const { cost_price, sale_price_latli } = req.body;
        const stockId = req.params.id;
        
        if (!cost_price && !sale_price_latli) {
            return res.status(400).json({
                success: false,
                message: 'Either cost_price or sale_price_latli must be provided'
            });
        }
        
        // Get current stock data
        const currentStock = await query('SELECT * FROM tbl_stock WHERE id = ?', [stockId]);
        if (currentStock.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Stock item not found'
            });
        }
        
        const oldCostPrice = currentStock[0].cost_price || 0;
        const oldSalePrice = currentStock[0].sale_price_latli || 0;
        const newCostPrice = cost_price !== undefined ? cost_price : oldCostPrice;
        const newSalePrice = sale_price_latli !== undefined ? sale_price_latli : oldSalePrice;
        
        // Track price change
        if (oldCostPrice !== newCostPrice || oldSalePrice !== newSalePrice) {
            await stockUtils.trackPriceChange(
                stockId,
                oldCostPrice,
                newCostPrice,
                oldSalePrice,
                newSalePrice,
                req.user.name || req.user.username || 'System'
            );
        }
        
        // Update stock price
        const updatedStock = await stockUtils.updateStockPrice(stockId, newCostPrice, newSalePrice);
        
        res.json({
            success: true,
            message: 'Stock price updated successfully',
            data: updatedStock
        });
    } catch (error) {
        next(error);
    }
});

// Get price history for a stock item
router.get('/:id/price-history', async (req, res, next) => {
    try {
        const stockId = req.params.id;
        const limit = parseInt(req.query.limit) || 50;
        
        const history = await stockUtils.getPriceHistory(stockId, limit);
        
        res.json({
            success: true,
            count: history.length,
            data: history
        });
    } catch (error) {
        next(error);
    }
});

module.exports = router;

