const express = require('express');
const router = express.Router();
const { query } = require('../config/database');
const { protect } = require('../middleware/auth');

// Get all vitals for a patient
router.get('/patient/:patientId', protect, async (req, res) => {
    try {
        const vitals = await query(
            `SELECT * FROM tbl_patient_vitals 
             WHERE patient_id = ? 
             ORDER BY recorded_date DESC`,
            [req.params.patientId]
        );
        res.json({ success: true, data: vitals });
    } catch (error) {
        res.status(500).json({ success: false, message: error.message });
    }
});

// Get all patients with their latest vitals
router.get('/patients-list', protect, async (req, res) => {
    try {
        // Use subquery to get latest vitals per patient (include patients without vitals)
        const patients = await query(
            `SELECT 
                p.patient_id,
                p.name,
                p.age,
                p.phone,
                p.address,
                p.case_,
                p.dr_name,
                p.room_no,
                v.blood_pressure_systolic as bp_sys,
                v.blood_pressure_diastolic as bp_dia,
                v.heart_rate,
                v.temperature,
                v.respiratory_rate,
                v.oxygen_saturation as spo2,
                v.weight,
                v.blood_glucose as glucose,
                v.recorded_date as last_vital_date,
                v.recorded_by
            FROM tbl_patient p
            LEFT JOIN (
                SELECT v1.*
                FROM tbl_patient_vitals v1
                INNER JOIN (
                    SELECT patient_id, MAX(recorded_date) as max_date
                    FROM tbl_patient_vitals
                    GROUP BY patient_id
                ) v2 ON v1.patient_id = v2.patient_id AND v1.recorded_date = v2.max_date
            ) v ON p.patient_id = v.patient_id
            ORDER BY 
                COALESCE(v.recorded_date, '1900-01-01') DESC,
                p.name ASC
            LIMIT 500`
        );
        res.json({ success: true, data: patients });
    } catch (error) {
        res.status(500).json({ success: false, message: error.message });
    }
});

// Get vitals chart data
router.get('/patient/:patientId/chart', protect, async (req, res) => {
    try {
        const vitals = await query(
            `SELECT 
                DATE_FORMAT(recorded_date, '%Y-%m-%d %H:%i') as date,
                blood_pressure_systolic as bp_sys,
                blood_pressure_diastolic as bp_dia,
                heart_rate,
                temperature,
                respiratory_rate,
                oxygen_saturation as spo2,
                weight,
                height,
                bmi,
                blood_glucose as glucose
             FROM tbl_patient_vitals 
             WHERE patient_id = ? 
             ORDER BY recorded_date ASC
             LIMIT 100`,
            [req.params.patientId]
        );
        res.json({ success: true, data: vitals });
    } catch (error) {
        res.status(500).json({ success: false, message: error.message });
    }
});

// Add new vital signs
router.post('/', protect, async (req, res) => {
    try {
        const {
            patient_id,
            blood_pressure_systolic,
            blood_pressure_diastolic,
            heart_rate,
            temperature,
            respiratory_rate,
            oxygen_saturation,
            weight,
            height,
            blood_glucose,
            notes
        } = req.body;

        // Calculate BMI if height and weight provided
        let bmi = null;
        if (weight && height && height > 0) {
            bmi = (weight / ((height / 100) ** 2)).toFixed(2);
        }

        // Helper function to convert undefined to null (mysql2 requires null, not undefined)
        const toNull = (value) => value === undefined ? null : value;
        
        const result = await query(
            `INSERT INTO tbl_patient_vitals 
            (patient_id, blood_pressure_systolic, blood_pressure_diastolic, heart_rate, 
             temperature, respiratory_rate, oxygen_saturation, weight, height, bmi, 
             blood_glucose, recorded_by, notes) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
            [
                toNull(patient_id),
                toNull(blood_pressure_systolic),
                toNull(blood_pressure_diastolic),
                toNull(heart_rate),
                toNull(temperature),
                toNull(respiratory_rate),
                toNull(oxygen_saturation),
                toNull(weight),
                toNull(height),
                bmi, // already null if not calculated
                toNull(blood_glucose),
                req.user.name, // this should always be defined from auth
                toNull(notes)
            ]
        );

        res.status(201).json({
            success: true,
            message: 'Vital signs recorded',
            id: result.insertId
        });
    } catch (error) {
        res.status(500).json({ success: false, message: error.message });
    }
});

// Delete vital record
router.delete('/:id', protect, async (req, res) => {
    try {
        await query('DELETE FROM tbl_patient_vitals WHERE id = ?', [req.params.id]);
        res.json({ success: true, message: 'Vital record deleted' });
    } catch (error) {
        res.status(500).json({ success: false, message: error.message });
    }
});

module.exports = router;

