const express = require('express');
const router = express.Router();
const { protect } = require('../middleware/auth');
const { requireCrudPermission } = require('../middleware/permissions');
const warehouseController = require('../controllers/warehouse.controller');
const transferController = require('../controllers/warehouse-transfer.controller');
const movementController = require('../controllers/warehouse-movement.controller');

router.use(protect);
router.use(requireCrudPermission('warehouse'));

// Warehouse CRUD
router.route('/')
    .get(warehouseController.getWarehouses)
    .post(warehouseController.createWarehouse);

// Stock Transfers - MUST be before /:id route
router.route('/transfers')
    .get(transferController.getTransfers)
    .post(transferController.createTransfer);

router.route('/transfers/:id')
    .get(transferController.getTransfer);

router.route('/transfers/:id/complete')
    .put(transferController.completeTransfer);

router.route('/transfers/:id/cancel')
    .put(transferController.cancelTransfer);

// Stock Movements - MUST be before /:id route
router.route('/movements')
    .get(movementController.getMovements);

router.route('/movements/summary')
    .get(movementController.getMovementSummary);

router.route('/movements/by-person')
    .get(movementController.getMovementsByPerson);

// Warehouse Stock - MUST be before /:id route
// More specific path first: .../stock/:warehouseStockId/quantity before /:id/stock
router.route('/:warehouseId/stock/:warehouseStockId/quantity')
    .put(warehouseController.updateWarehouseStockQuantity);

router.route('/:id/stock')
    .get(warehouseController.getWarehouseStock);

router.route('/:id/assign-stock')
    .post(warehouseController.assignStock);

// Warehouse CRUD by ID - MUST be last to avoid conflicts
router.route('/:id')
    .get(warehouseController.getWarehouse)
    .put(warehouseController.updateWarehouse)
    .delete(warehouseController.deleteWarehouse);

module.exports = router;

