/**
 * Complete Setup Script
 * Runs migrations and seeding in sequence
 */

require('dotenv').config();
const fs = require('fs');
const path = require('path');
const { pool, testConnection } = require('./config/database');

async function runFullSetup() {
    console.log('🚀 Starting Complete Database Setup\n');
    console.log('═══════════════════════════════════════════════════════\n');

    try {
        // Step 1: Test connection
        console.log('Step 1: Testing Database Connection...');
        const connected = await testConnection();
        if (!connected) {
            throw new Error('Database connection failed');
        }
        console.log('✅ Connected to MySQL\n');

        // Step 2: Run main schema
        console.log('Step 2: Creating Database Schema...');
        await runSQLFile('../database/infotech_db_mysql.sql');
        console.log('✅ Main schema created\n');

        // Step 3: Add image columns
        console.log('Step 3: Adding Image Support...');
        await runSQLFile('../database/add_image_column.sql');
        console.log('✅ Image columns added\n');

        // Step 4: Update EMR schema
        console.log('Step 4: Updating EMR Schema...');
        await runSQLFile('../database/update_emr_schema.sql');
        console.log('✅ EMR schema updated\n');

        // Step 5: Update shop info schema
        console.log('Step 5: Updating Shop Info Schema...');
        await runSQLFile('../database/update_shopinfo_schema.sql');
        console.log('✅ Shop info schema updated\n');

        // Step 6: Run complete schema
        console.log('Step 6: Running Complete Schema Update...');
        await runSQLFile('../database/complete_schema.sql');
        console.log('✅ Complete schema applied\n');

        // Step 7: Seed database
        console.log('Step 7: Seeding Initial Data...');
        const { seedDatabase } = require('./utils/seed');
        await seedDatabase();
        console.log('✅ Database seeded\n');

        // Step 8: Verification
        console.log('Step 8: Verifying Setup...');
        const tables = await pool.query('SHOW TABLES');
        console.log(`✅ Total tables created: ${tables[0].length}\n`);

        console.log('═══════════════════════════════════════════════════════');
        console.log('🎉 SETUP COMPLETED SUCCESSFULLY!');
        console.log('═══════════════════════════════════════════════════════\n');
        
        console.log('📝 Next Steps:');
        console.log('   1. Run: npm run verify (to verify all tables)');
        console.log('   2. Run: npm start (to start the server)');
        console.log('   3. Open: frontend/index.html (to access UI)');
        console.log('   4. Login: admin / admin\n');

        process.exit(0);

    } catch (error) {
        console.error('\n❌ SETUP FAILED:', error.message);
        console.error(error);
        process.exit(1);
    }
}

async function runSQLFile(relativePath) {
    const sqlPath = path.join(__dirname, relativePath);
    
    if (!fs.existsSync(sqlPath)) {
        console.log(`   ⚠️  File not found: ${sqlPath}`);
        return;
    }

    const sql = fs.readFileSync(sqlPath, 'utf8');
    const statements = sql
        .split(';')
        .map(s => s.trim())
        .filter(s => s.length > 0 && !s.startsWith('--'));

    const connection = await pool.getConnection();

    for (const statement of statements) {
        if (!statement || statement.startsWith('--') || statement.startsWith('/*')) {
            continue;
        }

        try {
            await connection.query(statement);
        } catch (error) {
            // Ignore "already exists" errors
            if (!error.message.includes('already exists') && 
                !error.message.includes('Duplicate') &&
                !error.message.includes('Unknown database')) {
                console.log(`   ⚠️  ${error.message.substring(0, 100)}`);
            }
        }
    }

    connection.release();
}

// Run setup
runFullSetup();

