const mysql = require('mysql2/promise');
const path = require('path');
require('dotenv').config({ path: path.join(__dirname, '.env') });

const dbConfig = {
    host: process.env.DB_HOST || 'localhost',
    user: process.env.DB_USER || 'root',
    password: process.env.DB_PASSWORD || '',
    database: process.env.DB_NAME || 'clinic_pro_db',
    port: process.env.DB_PORT || 3306,
    connectTimeout: 60000,
    acquireTimeout: 60000,
    timeout: 60000
};

async function runSupplierMigration() {
    let connection;
    try {
        console.log('🔗 Connecting to database...');
        connection = await mysql.createConnection(dbConfig);
        console.log('✅ Connected to database');

        console.log('➕ Adding email column to tbl_supplier...');
        await connection.execute(`
            ALTER TABLE tbl_supplier
            ADD COLUMN IF NOT EXISTS \`email\` VARCHAR(100) NULL AFTER \`ph_no\`;
        `);
        console.log('✅ Email column added successfully!');

        console.log('➕ Adding credit_balance column to tbl_supplier...');
        await connection.execute(`
            ALTER TABLE tbl_supplier
            ADD COLUMN IF NOT EXISTS \`credit_balance\` DECIMAL(10, 2) NOT NULL DEFAULT 0 AFTER \`address\`;
        `);
        console.log('✅ Credit balance column added successfully!');

        console.log('➕ Creating indexes...');
        await connection.execute(`
            CREATE INDEX IF NOT EXISTS \`idx_supplier_name\` ON \`tbl_supplier\`(\`name\`);
        `);
        await connection.execute(`
            CREATE INDEX IF NOT EXISTS \`idx_supplier_ph_no\` ON \`tbl_supplier\`(\`ph_no\`);
        `);
        await connection.execute(`
            CREATE INDEX IF NOT EXISTS \`idx_supplier_email\` ON \`tbl_supplier\`(\`email\`);
        `);
        console.log('✅ Indexes created successfully!');

        // Verify the columns were added
        const [rows] = await connection.execute(`
            SELECT COLUMN_NAME, DATA_TYPE, IS_NULLABLE, COLUMN_DEFAULT
            FROM INFORMATION_SCHEMA.COLUMNS
            WHERE TABLE_SCHEMA = ? AND TABLE_NAME = 'tbl_supplier'
            ORDER BY ORDINAL_POSITION;
        `, [dbConfig.database]);
        console.log('\n📊 Final table structure:');
        console.table(rows);

        console.log('\n✅ Migration completed successfully!');

    } catch (error) {
        console.error('❌ Error during migration:', error);
    } finally {
        if (connection) {
            await connection.end();
            console.log('🔌 Database connection closed');
        }
    }
}

runSupplierMigration();

