/**
 * Script to add i18n.js to all HTML pages in frontend/pages
 * This ensures all pages have translation support
 */

const fs = require('fs');
const path = require('path');

const pagesDir = path.join(__dirname, '../../frontend/pages');
const i18nScript = '<script src="../js/i18n.js"></script>';

// Pages that already have i18n.js
const pagesWithI18n = ['dashboard.html', 'pos.html', 'discharged-patient-report.html', 'patients.html'];

function addI18nToPage(filePath) {
    try {
        let content = fs.readFileSync(filePath, 'utf8');
        const fileName = path.basename(filePath);
        
        // Skip if already has i18n.js
        if (content.includes('i18n.js')) {
            console.log(`✓ ${fileName} already has i18n.js`);
            return;
        }
        
        // Skip if in the list of pages that already have it
        if (pagesWithI18n.includes(fileName)) {
            console.log(`✓ ${fileName} already processed`);
            return;
        }
        
        // Find where to insert the script (after api.js or auth-check.js)
        const apiJsPattern = /<script src="\.\.\/js\/api\.js"><\/script>/;
        const authCheckPattern = /<script src="\.\.\/js\/auth-check\.js"><\/script>/;
        
        if (content.match(apiJsPattern)) {
            content = content.replace(apiJsPattern, `$&\n    ${i18nScript}`);
            fs.writeFileSync(filePath, content, 'utf8');
            console.log(`✓ Added i18n.js to ${fileName}`);
        } else if (content.match(authCheckPattern)) {
            content = content.replace(authCheckPattern, `$&\n    ${i18nScript}`);
            fs.writeFileSync(filePath, content, 'utf8');
            console.log(`✓ Added i18n.js to ${fileName}`);
        } else {
            // Try to find any script tag and add after it
            const scriptPattern = /(<script src="[^"]+"><\/script>)/;
            if (content.match(scriptPattern)) {
                content = content.replace(scriptPattern, `$1\n    ${i18nScript}`);
                fs.writeFileSync(filePath, content, 'utf8');
                console.log(`✓ Added i18n.js to ${fileName} (after first script)`);
            } else {
                console.log(`⚠ Could not find insertion point in ${fileName}`);
            }
        }
    } catch (error) {
        console.error(`✗ Error processing ${filePath}:`, error.message);
    }
}

// Get all HTML files
const files = fs.readdirSync(pagesDir).filter(file => file.endsWith('.html'));

console.log(`Found ${files.length} HTML files in frontend/pages\n`);

files.forEach(file => {
    const filePath = path.join(pagesDir, file);
    addI18nToPage(filePath);
});

console.log('\n✓ Done! All pages should now have i18n.js');

