/**
 * Script to add unit and reference_ranges columns to tbl_laboratory_item
 * Run this if the columns are missing from the database
 */

require('dotenv').config({ path: require('path').join(__dirname, '../.env') });
const { query } = require('../config/database');

async function addLabItemColumns() {
    try {
        console.log('🔍 Checking for unit and reference_ranges columns in tbl_laboratory_item...\n');

        // Check if unit column exists
        const unitCheck = await query(`
            SELECT COUNT(*) as count 
            FROM INFORMATION_SCHEMA.COLUMNS 
            WHERE TABLE_SCHEMA = DATABASE() 
            AND TABLE_NAME = 'tbl_laboratory_item' 
            AND COLUMN_NAME = 'unit'
        `);

        if (unitCheck[0].count === 0) {
            console.log('⚠️  unit column not found, adding...');
            await query(`
                ALTER TABLE tbl_laboratory_item 
                ADD COLUMN unit VARCHAR(50) NULL COMMENT 'Unit of measurement (e.g., mg/dL, mmol/L, g/L)' AFTER sale_price
            `);
            console.log('✅ unit column added successfully');
        } else {
            console.log('✅ unit column already exists');
        }

        // Check if reference_ranges column exists
        const refCheck = await query(`
            SELECT COUNT(*) as count 
            FROM INFORMATION_SCHEMA.COLUMNS 
            WHERE TABLE_SCHEMA = DATABASE() 
            AND TABLE_NAME = 'tbl_laboratory_item' 
            AND COLUMN_NAME = 'reference_ranges'
        `);

        if (refCheck[0].count === 0) {
            console.log('⚠️  reference_ranges column not found, adding...');
            await query(`
                ALTER TABLE tbl_laboratory_item 
                ADD COLUMN reference_ranges TEXT NULL COMMENT 'Normal reference ranges for the test (e.g., "Male: 3.5-5.5, Female: 3.5-5.0")' AFTER unit
            `);
            console.log('✅ reference_ranges column added successfully');
        } else {
            console.log('✅ reference_ranges column already exists');
        }

        // Check if index exists
        const indexCheck = await query(`
            SELECT COUNT(*) as count 
            FROM INFORMATION_SCHEMA.STATISTICS 
            WHERE TABLE_SCHEMA = DATABASE() 
            AND TABLE_NAME = 'tbl_laboratory_item' 
            AND INDEX_NAME = 'idx_unit'
        `);

        if (indexCheck[0].count === 0) {
            console.log('⚠️  idx_unit index not found, adding...');
            await query(`
                CREATE INDEX idx_unit ON tbl_laboratory_item(unit)
            `);
            console.log('✅ idx_unit index added successfully');
        } else {
            console.log('✅ idx_unit index already exists');
        }

        console.log('\n✅ All columns and indexes are in place!');
        console.log('You can now save and view Unit and Reference Ranges in Laboratory Items.\n');

    } catch (error) {
        console.error('❌ Error adding columns:', error);
        process.exit(1);
    }
}

// Run the script
addLabItemColumns()
    .then(() => {
        console.log('Script completed successfully');
        process.exit(0);
    })
    .catch((error) => {
        console.error('Script failed:', error);
        process.exit(1);
    });
