/**
 * Create Database Script
 * Creates the database if it doesn't exist
 */

require('dotenv').config({ path: require('path').join(__dirname, '../.env') });
const mysql = require('mysql2/promise');

async function createDatabase() {
    const dbName = process.env.DB_NAME || 'clinic_pro_db';
    const dbHost = process.env.DB_HOST || '127.0.0.1';
    const dbPort = parseInt(process.env.DB_PORT) || 3306;
    const dbUser = process.env.DB_USER || 'root';
    const dbPassword = process.env.DB_PASSWORD || '';

    console.log('🔧 Creating Database...\n');
    console.log(`Database Name: ${dbName}`);
    console.log(`Host: ${dbHost}:${dbPort}`);
    console.log(`User: ${dbUser}\n`);

    try {
        // Connect to MySQL without specifying a database
        const connection = await mysql.createConnection({
            host: dbHost,
            port: dbPort,
            user: dbUser,
            password: dbPassword,
            // Don't specify database here
        });

        console.log('✅ Connected to MySQL server\n');

        // Create database if it doesn't exist
        const [databases] = await connection.query(
            `SELECT SCHEMA_NAME FROM INFORMATION_SCHEMA.SCHEMATA WHERE SCHEMA_NAME = ?`,
            [dbName]
        );

        if (databases.length === 0) {
            console.log(`📦 Creating database '${dbName}'...`);
            await connection.query(
                `CREATE DATABASE \`${dbName}\` DEFAULT CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci`
            );
            console.log(`✅ Database '${dbName}' created successfully!\n`);
        } else {
            console.log(`ℹ️  Database '${dbName}' already exists.\n`);
        }

        await connection.end();
        console.log('✅ Database setup complete!\n');
        return true;
    } catch (error) {
        console.error('❌ Error creating database:', error.message);
        console.error('\nPlease check:');
        console.error('1. MySQL server is running');
        console.error('2. Database credentials in .env file are correct');
        console.error('3. User has CREATE DATABASE permission\n');
        return false;
    }
}

// Run if called directly
if (require.main === module) {
    createDatabase()
        .then((success) => {
            process.exit(success ? 0 : 1);
        })
        .catch((error) => {
            console.error('Fatal error:', error);
            process.exit(1);
        });
}

module.exports = { createDatabase };

