#!/usr/bin/env node

/**
 * Permission Diagnostics Script
 * 
 * Diagnoses permission issues for users and roles
 * Usage: node scripts/diagnose-permissions.js [userId]
 */

const path = require('path');
require('dotenv').config({ path: path.join(__dirname, '../.env') });

const { sequelize, User, Role, Permission, RolePermission } = require('../models/sequelize');
const { getUserPermissions, hasPermission } = require('../utils/permissions');

async function diagnoseUser(userId) {
    console.log(`\n🔍 Diagnosing permissions for User ID: ${userId}\n`);
    console.log('='.repeat(60));

    try {
        // Get user with role
        const user = await User.findByPk(userId, {
            include: [{
                model: Role,
                as: 'role',
                include: [{
                    model: Permission,
                    as: 'permissions',
                    through: {
                        attributes: ['granted', 'active']
                    }
                }]
            }]
        });

        if (!user) {
            console.error(`❌ User with ID ${userId} not found`);
            return;
        }

        console.log(`\n👤 User Information:`);
        console.log(`   ID: ${user.id}`);
        console.log(`   Name: ${user.name}`);
        console.log(`   Role ID: ${user.roleId}`);

        if (!user.role) {
            console.error(`\n❌ User has no role assigned!`);
            return;
        }

        console.log(`\n🎭 Role Information:`);
        console.log(`   ID: ${user.role.id}`);
        console.log(`   Name: ${user.role.name}`);
        console.log(`   Level: ${user.role.level}`);
        console.log(`   Active: ${user.role.active}`);
        console.log(`   Description: ${user.role.description || 'N/A'}`);

        const metadata = typeof user.role.metadata === 'string'
            ? JSON.parse(user.role.metadata)
            : user.role.metadata || {};

        if (metadata.bypassChecks || user.role.level >= 100) {
            console.log(`   ⚡ Admin Bypass: ENABLED`);
        }

        // Get all permissions
        const allPermissions = await Permission.findAll({
            where: { active: true },
            order: [['resource', 'ASC'], ['action', 'ASC']]
        });

        console.log(`\n📋 Permission Summary:`);
        console.log(`   Total Permissions: ${allPermissions.length}`);
        console.log(`   Assigned Permissions: ${user.role.permissions?.length || 0}`);

        // Get user permissions
        const userPerms = await getUserPermissions(userId);

        console.log(`\n✅ User Permissions:`);
        const resources = Object.keys(userPerms).sort();
        let totalGranted = 0;

        for (const resource of resources) {
            const actions = Object.keys(userPerms[resource]).filter(
                action => userPerms[resource][action] === true
            );
            if (actions.length > 0) {
                console.log(`   ${resource}: ${actions.join(', ')}`);
                totalGranted += actions.length;
            }
        }

        console.log(`\n   Total Granted: ${totalGranted} permissions`);

        // Test specific permissions
        console.log(`\n🧪 Testing Common Permissions:`);
        const testPerms = [
            { resource: 'users', action: 'create' },
            { resource: 'users', action: 'read' },
            { resource: 'patients', action: 'create' },
            { resource: 'patients', action: 'read' },
            { resource: 'reports', action: 'read' },
            { resource: 'settings', action: 'manage' }
        ];

        for (const perm of testPerms) {
            const has = await hasPermission(userId, perm.resource, perm.action);
            const icon = has ? '✅' : '❌';
            console.log(`   ${icon} ${perm.resource}:${perm.action}`);
        }

        // Check for missing role-permission assignments
        console.log(`\n🔗 Role-Permission Assignments:`);
        const rolePerms = await RolePermission.findAll({
            where: {
                role_id: user.role.id,
                active: true
            },
            include: [{
                model: Permission,
                as: 'permission'
            }]
        });

        console.log(`   Total Assignments: ${rolePerms.length}`);
        console.log(`   Granted: ${rolePerms.filter(rp => rp.granted).length}`);
        console.log(`   Denied: ${rolePerms.filter(rp => !rp.granted).length}`);

    } catch (error) {
        console.error(`\n❌ Error: ${error.message}`);
        console.error(error.stack);
    }
}

async function diagnoseAll() {
    console.log(`\n🔍 Diagnosing All Roles and Permissions\n`);
    console.log('='.repeat(60));

    try {
        // List all roles
        const roles = await Role.findAll({
            include: [{
                model: Permission,
                as: 'permissions',
                through: {
                    attributes: ['granted', 'active']
                }
            }],
            order: [['level', 'DESC']]
        });

        console.log(`\n🎭 Roles (${roles.length}):`);
        for (const role of roles) {
            const userCount = await User.count({ where: { roleId: role.id } });
            console.log(`\n   ${role.name} (Level ${role.level})`);
            console.log(`      Active: ${role.active}`);
            console.log(`      Users: ${userCount}`);
            console.log(`      Permissions: ${role.permissions?.length || 0}`);
        }

        // List all permissions
        const permissions = await Permission.findAll({
            where: { active: true },
            order: [['resource', 'ASC'], ['action', 'ASC']]
        });

        console.log(`\n📋 Permissions (${permissions.length}):`);
        const byResource = {};
        for (const perm of permissions) {
            if (!byResource[perm.resource]) {
                byResource[perm.resource] = [];
            }
            byResource[perm.resource].push(perm.action);
        }

        for (const resource of Object.keys(byResource).sort()) {
            console.log(`   ${resource}: ${byResource[resource].join(', ')}`);
        }

    } catch (error) {
        console.error(`\n❌ Error: ${error.message}`);
        console.error(error.stack);
    }
}

async function main() {
    const userId = process.argv[2];

    try {
        // Test connection
        await sequelize.authenticate();
        console.log('✅ Database connection established');

        if (userId) {
            await diagnoseUser(parseInt(userId));
        } else {
            await diagnoseAll();
        }

    } catch (error) {
        console.error('❌ Database connection failed:', error.message);
        process.exit(1);
    } finally {
        await sequelize.close();
    }
}

// Run if called directly
if (require.main === module) {
    main().catch(console.error);
}

module.exports = { diagnoseUser, diagnoseAll };

