#!/usr/bin/env node
/**
 * Ensure stockExpense permissions exist and grant to all roles.
 *
 * Usage:
 *   node backend/scripts/ensure-stock-expense-permissions.js
 */

const path = require('path');
require('dotenv').config({ path: path.join(__dirname, '../.env') });

const { sequelize, Role, Permission, RolePermission } = require('../models/sequelize');

const actions = ['create', 'read', 'update', 'delete', 'export'];

async function ensureStockExpensePermissions() {
    console.log('🔐 Ensuring stockExpense permissions...');

    // Ensure permissions exist
    const permissions = [];
    for (const action of actions) {
        const [perm] = await Permission.findOrCreate({
            where: { resource: 'stockExpense', action },
            defaults: {
                resource: 'stockExpense',
                action,
                description: `${action.charAt(0).toUpperCase() + action.slice(1)} stockExpense`,
                active: true,
                metadata: { category: 'stockExpense' }
            }
        });
        if (!perm.active) {
            await perm.update({ active: true });
        }
        permissions.push(perm);
    }

    // Grant permissions to all roles
    const roles = await Role.findAll();
    const now = new Date();
    const rolePerms = [];

    for (const role of roles) {
        for (const perm of permissions) {
            rolePerms.push({
                roleId: role.id,
                permissionId: perm.id,
                granted: true,
                active: true,
                metadata: {},
                created_at: now,
                updated_at: now
            });
        }
    }

    await RolePermission.bulkCreate(rolePerms, {
        updateOnDuplicate: ['granted', 'active', 'updated_at']
    });

    console.log(`✅ Granted ${permissions.length} stockExpense permissions to ${roles.length} roles`);
}

ensureStockExpensePermissions()
    .then(async () => {
        await sequelize.close();
        console.log('🎉 stockExpense permissions ensured.');
        process.exit(0);
    })
    .catch(async (error) => {
        console.error('❌ Failed to ensure stockExpense permissions:', error);
        try {
            await sequelize.close();
        } catch (e) {}
        process.exit(1);
    });
