/**
 * Fresh Installation Script
 * Runs all necessary setup commands for a new installation
 * 
 * Usage: node backend/scripts/fresh-install.js
 */

const { execSync } = require('child_process');
const path = require('path');
const fs = require('fs');

console.log('🚀 Starting Fresh Installation...\n');

// Change to project root
const projectRoot = path.join(__dirname, '../..');
process.chdir(projectRoot);

// Check if .env exists
const envPath = path.join(projectRoot, 'backend', '.env');
if (!fs.existsSync(envPath)) {
    console.log('⚠️  WARNING: .env file not found!');
    console.log('   Please create backend/.env with your database credentials.\n');
    console.log('   Required variables:');
    console.log('   - DB_HOST');
    console.log('   - DB_PORT');
    console.log('   - DB_USER');
    console.log('   - DB_PASSWORD');
    console.log('   - DB_NAME');
    console.log('   - JWT_SECRET\n');
    process.exit(1);
}

const steps = [
    {
        name: 'Step 1: Create Main Tables',
        command: 'node backend/utils/create-all-tables.js',
        description: 'Creating all base tables (97+ tables)...'
    },
    {
        name: 'Step 2: Run SQL Migrations',
        command: 'node backend/migrations/run-create-all-tables.js',
        description: 'Running incremental migrations...'
    },
    {
        name: 'Step 3: Create Missing Tables',
        command: 'node backend/migrations/create-missing-tables.js',
        description: 'Ensuring all tables exist...'
    },
    {
        name: 'Step 4: Verify Setup',
        command: 'node backend/migrations/verify-setup.js',
        description: 'Verifying database setup...'
    }
];

let successCount = 0;
let failCount = 0;

for (const step of steps) {
    try {
        console.log(`\n${'='.repeat(60)}`);
        console.log(`📦 ${step.name}`);
        console.log(`   ${step.description}`);
        console.log(`${'='.repeat(60)}\n`);
        
        execSync(step.command, { 
            stdio: 'inherit',
            cwd: projectRoot,
            env: { ...process.env, NODE_ENV: 'development' }
        });
        
        successCount++;
        console.log(`\n✅ ${step.name} completed successfully!\n`);
    } catch (error) {
        failCount++;
        console.error(`\n❌ ${step.name} failed!`);
        console.error(`   Error: ${error.message}\n`);
        
        // Ask if user wants to continue
        console.log('⚠️  Continuing with next step...\n');
    }
}

// Summary
console.log('\n' + '='.repeat(60));
console.log('📊 INSTALLATION SUMMARY');
console.log('='.repeat(60));
console.log(`✅ Successful: ${successCount}/${steps.length}`);
console.log(`❌ Failed: ${failCount}/${steps.length}\n`);

if (failCount === 0) {
    console.log('🎉 Fresh installation completed successfully!\n');
    console.log('Next steps:');
    console.log('   1. (Optional) Seed initial data:');
    console.log('      cd backend && node seed-hospital-fixed.js\n');
    console.log('   2. Start the server:');
    console.log('      npm run dev\n');
} else {
    console.log('⚠️  Installation completed with errors.\n');
    console.log('Please review the errors above and:');
    console.log('   1. Check your database connection in .env');
    console.log('   2. Ensure database exists');
    console.log('   3. Verify database credentials\n');
    console.log('For help, see: FRESH_INSTALLATION_GUIDE.md\n');
}

process.exit(failCount > 0 ? 1 : 0);

