#!/usr/bin/env node

/**
 * List Permissions Script
 * 
 * Lists all permissions, roles, and their assignments
 * Usage: node scripts/list-permissions.js [--role=roleName] [--resource=resourceName]
 */

const path = require('path');
require('dotenv').config({ path: path.join(__dirname, '../.env') });

const { sequelize, Role, Permission, RolePermission } = require('../models/sequelize');

async function listAllPermissions() {
    console.log('\n📋 All Permissions\n');
    console.log('='.repeat(60));

    try {
        const permissions = await Permission.findAll({
            where: { active: true },
            order: [['resource', 'ASC'], ['action', 'ASC']]
        });

        const byResource = {};
        for (const perm of permissions) {
            if (!byResource[perm.resource]) {
                byResource[perm.resource] = [];
            }
            byResource[perm.resource].push(perm.action);
        }

        for (const resource of Object.keys(byResource).sort()) {
            console.log(`\n${resource}:`);
            console.log(`  ${byResource[resource].join(', ')}`);
        }

        console.log(`\nTotal: ${permissions.length} permissions across ${Object.keys(byResource).length} resources`);
    } catch (error) {
        console.error(`❌ Error: ${error.message}`);
    }
}

async function listRoles() {
    console.log('\n🎭 All Roles\n');
    console.log('='.repeat(60));

    try {
        const roles = await Role.findAll({
            include: [{
                model: Permission,
                as: 'permissions',
                through: {
                    where: { active: true, granted: true },
                    attributes: []
                }
            }],
            order: [['level', 'DESC']]
        });

        for (const role of roles) {
            console.log(`\n${role.name} (Level ${role.level})`);
            console.log(`  Active: ${role.active ? 'Yes' : 'No'}`);
            console.log(`  Description: ${role.description || 'N/A'}`);
            console.log(`  Permissions: ${role.permissions?.length || 0}`);
        }
    } catch (error) {
        console.error(`❌ Error: ${error.message}`);
    }
}

async function listRolePermissions(roleName) {
    console.log(`\n🔗 Permissions for Role: ${roleName}\n`);
    console.log('='.repeat(60));

    try {
        const role = await Role.findOne({
            where: { name: roleName },
            include: [{
                model: Permission,
                as: 'permissions',
                through: {
                    where: { active: true, granted: true },
                    attributes: ['granted', 'active']
                }
            }]
        });

        if (!role) {
            console.error(`❌ Role '${roleName}' not found`);
            return;
        }

        console.log(`Role: ${role.name} (Level ${role.level})`);
        console.log(`Active: ${role.active ? 'Yes' : 'No'}`);

        const byResource = {};
        for (const perm of role.permissions || []) {
            if (!byResource[perm.resource]) {
                byResource[perm.resource] = [];
            }
            byResource[perm.resource].push(perm.action);
        }

        console.log(`\nPermissions (${role.permissions?.length || 0}):`);
        for (const resource of Object.keys(byResource).sort()) {
            console.log(`  ${resource}: ${byResource[resource].join(', ')}`);
        }
    } catch (error) {
        console.error(`❌ Error: ${error.message}`);
    }
}

async function listResourcePermissions(resourceName) {
    console.log(`\n📦 Permissions for Resource: ${resourceName}\n`);
    console.log('='.repeat(60));

    try {
        const permissions = await Permission.findAll({
            where: {
                resource: resourceName,
                active: true
            },
            include: [{
                model: Role,
                as: 'roles',
                through: {
                    where: { active: true, granted: true },
                    attributes: []
                }
            }],
            order: [['action', 'ASC']]
        });

        if (permissions.length === 0) {
            console.log(`No permissions found for resource '${resourceName}'`);
            return;
        }

        for (const perm of permissions) {
            console.log(`\n${perm.resource}:${perm.action}`);
            console.log(`  Description: ${perm.description || 'N/A'}`);
            console.log(`  Roles with this permission: ${perm.roles?.length || 0}`);
            if (perm.roles && perm.roles.length > 0) {
                const roleNames = perm.roles.map(r => r.name).join(', ');
                console.log(`  ${roleNames}`);
            }
        }
    } catch (error) {
        console.error(`❌ Error: ${error.message}`);
    }
}

async function main() {
    const args = process.argv.slice(2);
    const roleArg = args.find(arg => arg.startsWith('--role='));
    const resourceArg = args.find(arg => arg.startsWith('--resource='));

    try {
        await sequelize.authenticate();
        console.log('✅ Database connection established');

        if (roleArg) {
            const roleName = roleArg.split('=')[1];
            await listRolePermissions(roleName);
        } else if (resourceArg) {
            const resourceName = resourceArg.split('=')[1];
            await listResourcePermissions(resourceName);
        } else {
            await listAllPermissions();
            await listRoles();
        }

    } catch (error) {
        console.error('❌ Error:', error.message);
        process.exit(1);
    } finally {
        await sequelize.close();
    }
}

// Run if called directly
if (require.main === module) {
    main().catch(console.error);
}

module.exports = {
    listAllPermissions,
    listRoles,
    listRolePermissions,
    listResourcePermissions
};

