#!/usr/bin/env node

/**
 * Permission Repair Script
 * 
 * Repairs common permission issues:
 * - Removes orphaned role-permission assignments
 * - Reassigns default permissions to roles
 * - Clears stale cache entries
 * 
 * Usage: node scripts/repair-permissions.js
 */

const path = require('path');
require('dotenv').config({ path: path.join(__dirname, '../.env') });

const { sequelize, Role, Permission, RolePermission, User } = require('../models/sequelize');
const { clearPermissionCache } = require('../utils/permissions');

async function repairOrphanedAssignments() {
    console.log('\n🔧 Repairing orphaned role-permission assignments...');

    try {
        // Find role-permission assignments where role or permission doesn't exist
        const [orphaned] = await sequelize.query(`
            SELECT rp.id, rp.role_id, rp.permission_id
            FROM tbl_role_permission rp
            LEFT JOIN tbl_role r ON rp.role_id = r.id
            LEFT JOIN tbl_permission p ON rp.permission_id = p.id
            WHERE r.id IS NULL OR p.id IS NULL
        `);

        if (orphaned.length > 0) {
            console.log(`   Found ${orphaned.length} orphaned assignments`);
            for (const orphan of orphaned) {
                await RolePermission.destroy({ where: { id: orphan.id } });
            }
            console.log(`   ✅ Removed ${orphaned.length} orphaned assignments`);
        } else {
            console.log('   ✅ No orphaned assignments found');
        }
    } catch (error) {
        console.error(`   ❌ Error: ${error.message}`);
    }
}

async function repairInactiveAssignments() {
    console.log('\n🔧 Checking inactive role-permission assignments...');

    try {
        // Find assignments where role or permission is inactive
        const [inactive] = await sequelize.query(`
            SELECT rp.id, rp.role_id, rp.permission_id, r.active as role_active, p.active as perm_active
            FROM tbl_role_permission rp
            JOIN tbl_role r ON rp.role_id = r.id
            JOIN tbl_permission p ON rp.permission_id = p.id
            WHERE (r.active = 0 OR p.active = 0) AND rp.active = 1
        `);

        if (inactive.length > 0) {
            console.log(`   Found ${inactive.length} assignments with inactive role/permission`);
            for (const item of inactive) {
                await RolePermission.update(
                    { active: false },
                    { where: { id: item.id } }
                );
            }
            console.log(`   ✅ Deactivated ${inactive.length} assignments`);
        } else {
            console.log('   ✅ No inactive assignments found');
        }
    } catch (error) {
        console.error(`   ❌ Error: ${error.message}`);
    }
}

async function verifyDefaultRoles() {
    console.log('\n🔧 Verifying default roles...');

    try {
        const defaultRoles = [
            { id: 1, name: 'Admin', level: 100 },
            { id: 2, name: 'Manager', level: 50 },
            { id: 3, name: 'Staff', level: 10 }
        ];

        for (const roleData of defaultRoles) {
            let role = await Role.findByPk(roleData.id);
            
            if (!role) {
                console.log(`   Creating missing role: ${roleData.name}`);
                role = await Role.create({
                    id: roleData.id,
                    name: roleData.name,
                    level: roleData.level,
                    active: true,
                    description: `Default ${roleData.name} role`
                });
            } else {
                // Update if needed
                if (role.name !== roleData.name || role.level !== roleData.level) {
                    console.log(`   Updating role: ${roleData.name}`);
                    await role.update({
                        name: roleData.name,
                        level: roleData.level
                    });
                }
            }
        }

        console.log('   ✅ Default roles verified');
    } catch (error) {
        console.error(`   ❌ Error: ${error.message}`);
    }
}

async function removeUnwantedPermissions() {
    console.log('\n🔧 Removing unwanted permissions (import, manage)...');

    try {
        // Find all 'import' and 'manage' permissions
        const [unwanted] = await sequelize.query(`
            SELECT id, resource, action
            FROM tbl_permission
            WHERE action IN ('import', 'manage')
        `);

        if (unwanted.length > 0) {
            console.log(`   Found ${unwanted.length} unwanted permissions to remove`);
            
            // First, remove role-permission assignments
            const permissionIds = unwanted.map(p => p.id);
            const [assignments] = await sequelize.query(`
                SELECT COUNT(*) as count
                FROM tbl_role_permission
                WHERE permission_id IN (${permissionIds.join(',')})
            `);
            
            if (assignments[0].count > 0) {
                await sequelize.query(`
                    DELETE FROM tbl_role_permission
                    WHERE permission_id IN (${permissionIds.join(',')})
                `);
                console.log(`   ✅ Removed ${assignments[0].count} role-permission assignments`);
            }
            
            // Then delete the permissions
            await sequelize.query(`
                DELETE FROM tbl_permission
                WHERE action IN ('import', 'manage')
            `);
            
            console.log(`   ✅ Removed ${unwanted.length} unwanted permissions`);
            
            // Show what was removed
            unwanted.forEach(p => {
                console.log(`      - ${p.resource}.${p.action} (ID: ${p.id})`);
            });
        } else {
            console.log('   ✅ No unwanted permissions found');
        }
    } catch (error) {
        console.error(`   ❌ Error: ${error.message}`);
    }
}

async function clearAllCaches() {
    console.log('\n🔧 Clearing permission caches...');

    try {
        clearPermissionCache(); // Clear all
        console.log('   ✅ All permission caches cleared');
    } catch (error) {
        console.error(`   ❌ Error: ${error.message}`);
    }
}

async function generateReport() {
    console.log('\n📊 Generating repair report...');

    try {
        const roleCount = await Role.count();
        const permissionCount = await Permission.count();
        const assignmentCount = await RolePermission.count();
        const userCount = await User.count();

        const [activeAssignments] = await sequelize.query(`
            SELECT COUNT(*) as count
            FROM tbl_role_permission rp
            JOIN tbl_role r ON rp.role_id = r.id
            JOIN tbl_permission p ON rp.permission_id = p.id
            WHERE rp.active = 1 AND r.active = 1 AND p.active = 1 AND rp.granted = 1
        `);

        console.log('\n   System Status:');
        console.log(`   Roles: ${roleCount}`);
        console.log(`   Permissions: ${permissionCount}`);
        console.log(`   Role-Permission Assignments: ${assignmentCount}`);
        console.log(`   Active Assignments: ${activeAssignments[0].count}`);
        console.log(`   Users: ${userCount}`);

    } catch (error) {
        console.error(`   ❌ Error: ${error.message}`);
    }
}

async function main() {
    console.log('🔧 Permission System Repair Tool');
    console.log('='.repeat(60));

    try {
        // Test connection
        await sequelize.authenticate();
        console.log('✅ Database connection established\n');

        // Run repairs
        await repairOrphanedAssignments();
        await repairInactiveAssignments();
        await removeUnwantedPermissions();
        await verifyDefaultRoles();
        await clearAllCaches();
        await generateReport();

        console.log('\n✅ Repair completed successfully!');

    } catch (error) {
        console.error('\n❌ Repair failed:', error.message);
        console.error(error.stack);
        process.exit(1);
    } finally {
        await sequelize.close();
    }
}

// Run if called directly
if (require.main === module) {
    main().catch(console.error);
}

module.exports = {
    repairOrphanedAssignments,
    repairInactiveAssignments,
    removeUnwantedPermissions,
    verifyDefaultRoles,
    clearAllCaches
};

