#!/usr/bin/env node
/**
 * Reset Admin role and grant full permissions to user 'admin'
 *
 * Usage:
 *   node backend/scripts/reset-admin-full-permissions.js
 */

const path = require('path');
require('dotenv').config({ path: path.join(__dirname, '../.env') });

const { sequelize, Role, Permission, RolePermission } = require('../models/sequelize');

async function resetAdminPermissions() {
    console.log('🔐 Resetting Admin role permissions...');

    // Ensure Admin role exists
    let adminRole = await Role.findOne({ where: { name: 'Admin' } });
    if (!adminRole) {
        adminRole = await Role.create({
            name: 'Admin',
            level: 100,
            active: true,
            description: 'Full system access with all permissions',
            metadata: { color: '#dc3545', icon: 'shield', bypassChecks: true }
        });
        console.log(`✅ Created Admin role (id=${adminRole.id})`);
    } else {
        await adminRole.update({
            level: 100,
            active: true,
            metadata: { color: '#dc3545', icon: 'shield', bypassChecks: true }
        });
        console.log(`✅ Updated Admin role (id=${adminRole.id})`);
    }

    // Assign user 'admin' to Admin role (case-insensitive)
    const [adminUsers] = await sequelize.query(
        `SELECT id, name FROM tbl_user WHERE LOWER(name) = 'admin' LIMIT 1`
    );
    if (adminUsers.length > 0) {
        const adminUser = adminUsers[0];
        await sequelize.query(
            `UPDATE tbl_user SET role_id = ? WHERE id = ?`,
            { replacements: [adminRole.id, adminUser.id] }
        );
        console.log(`✅ Assigned user '${adminUser.name}' (id=${adminUser.id}) to Admin role`);
    } else {
        console.log('⚠️  No user named "admin" found. Skipped user role assignment.');
    }

    // Grant all active permissions to Admin role
    const permissions = await Permission.findAll({ where: { active: true } });
    const now = new Date();

    const rolePerms = permissions.map((perm) => ({
        roleId: adminRole.id,
        permissionId: perm.id,
        granted: true,
        active: true,
        metadata: {},
        created_at: now,
        updated_at: now
    }));

    await RolePermission.bulkCreate(rolePerms, {
        updateOnDuplicate: ['granted', 'active', 'updated_at']
    });

    console.log(`✅ Granted ${permissions.length} permissions to Admin role`);
}

resetAdminPermissions()
    .then(async () => {
        await sequelize.close();
        console.log('🎉 Admin permissions reset complete.');
        process.exit(0);
    })
    .catch(async (error) => {
        console.error('❌ Failed to reset Admin permissions:', error);
        try {
            await sequelize.close();
        } catch (e) {}
        process.exit(1);
    });
