#!/usr/bin/env node

/**
 * Setup Permissions Script
 * 
 * Runs migrations and seeders to set up the permission system
 * Usage: node scripts/setup-permissions.js
 */

const path = require('path');
require('dotenv').config({ path: path.join(__dirname, '../.env') });

const { sequelize } = require('../config/sequelize');
const fs = require('fs');

async function runMigrations() {
    console.log('\n📦 Running migrations...');

    try {
        const migrationPath = path.join(__dirname, '../migrations');
        const migrationFile = path.join(migrationPath, '001-create-permission-tables.js');
        
        if (fs.existsSync(migrationFile)) {
            const migration = require(migrationFile);
            if (migration.up) {
                await migration.up(
                    sequelize.getQueryInterface(),
                    sequelize.constructor
                );
                console.log(`   ✅ 001-create-permission-tables.js`);
            }
        } else {
            console.log('   ⚠️  Migration file not found');
        }
        
        console.log(`✅ Migration executed`);
    } catch (error) {
        console.error(`❌ Migration error: ${error.message}`);
        throw error;
    }
}

async function runSeeders() {
    console.log('\n🌱 Running seeders...');

    try {
        const seederPath = path.join(__dirname, '../seeders');
        const seedFiles = fs.readdirSync(seederPath)
            .filter(file => file.endsWith('.js'))
            .sort();

        for (const file of seedFiles) {
            const seeder = require(path.join(seederPath, file));
            if (seeder.up) {
                await seeder.up(
                    sequelize.getQueryInterface(),
                    sequelize.constructor
                );
                console.log(`   ✅ ${file}`);
            }
        }

        console.log(`✅ ${seedFiles.length} seeder(s) executed`);
    } catch (error) {
        console.error(`❌ Seeder error: ${error.message}`);
        throw error;
    }
}

async function main() {
    console.log('🚀 Permission System Setup');
    console.log('='.repeat(60));

    try {
        await sequelize.authenticate();
        console.log('✅ Database connection established');

        await runMigrations();
        await runSeeders();

        console.log('\n✅ Permission system setup completed successfully!');
        console.log('\nNext steps:');
        console.log('  1. Run: node scripts/diagnose-permissions.js');
        console.log('  2. Test permissions in your routes');
        console.log('  3. Check documentation: backend/docs/PERMISSIONS_README.md');

    } catch (error) {
        console.error('\n❌ Setup failed:', error.message);
        console.error(error.stack);
        process.exit(1);
    } finally {
        await sequelize.close();
    }
}

// Run if called directly
if (require.main === module) {
    main().catch(console.error);
}

