/**
 * Seeder: Role Permissions
 * Assigns permissions to default roles
 */

'use strict';

module.exports = {
    async up(queryInterface, Sequelize) {
        // Get all permissions
        const [permissions] = await queryInterface.sequelize.query(
            'SELECT id, resource, action FROM tbl_permission WHERE active = true'
        );

        // Admin gets all permissions
        const adminPermissions = permissions.map(p => ({
            role_id: 1,
            permission_id: p.id,
            granted: true,
            active: true,
            metadata: JSON.stringify({}),
            created_at: new Date(),
            updated_at: new Date()
        }));

        // Manager gets most permissions except user management and system settings
        const managerPermissions = permissions
            .filter(p => !['users', 'settings'].includes(p.resource) || p.action === 'read')
            .map(p => ({
                role_id: 2,
                permission_id: p.id,
                granted: true,
                active: true,
                metadata: JSON.stringify({}),
                created_at: new Date(),
                updated_at: new Date()
            }));

        // Staff gets read/create/update for most resources, limited delete
        const staffPermissions = permissions
            .filter(p => {
                const restrictedResources = ['users', 'settings', 'reports'];
                const restrictedActions = ['delete'];
                
                if (restrictedResources.includes(p.resource) && restrictedActions.includes(p.action)) {
                    return false;
                }
                return ['read', 'create', 'update', 'export'].includes(p.action);
            })
            .map(p => ({
                role_id: 3,
                permission_id: p.id,
                granted: true,
                active: true,
                metadata: JSON.stringify({}),
                created_at: new Date(),
                updated_at: new Date()
            }));

        // Insert role permissions
        const allRolePermissions = [...adminPermissions, ...managerPermissions, ...staffPermissions];

        for (const rp of allRolePermissions) {
            const [existing] = await queryInterface.sequelize.query(
                `SELECT id FROM tbl_role_permission WHERE role_id = ${rp.role_id} AND permission_id = ${rp.permission_id}`
            );

            if (existing.length === 0) {
                await queryInterface.bulkInsert('tbl_role_permission', [rp]);
            }
        }
    },

    async down(queryInterface, Sequelize) {
        await queryInterface.bulkDelete('tbl_role_permission', {
            role_id: [1, 2, 3]
        });
    }
};

