/**
 * Seeder: Users and Permissions from Dump File
 * Seeds users, roles, permissions, and role-permissions based on clinic_pro_db.sql dump
 * 
 * This seeder matches the exact data from the SQL dump file
 */

'use strict';

const bcrypt = require('bcryptjs');

module.exports = {
    async up(queryInterface, Sequelize) {
        console.log('🌱 Seeding users, roles, permissions from dump file...');

        // 1. Seed Roles (5 roles from dump)
        const roles = [
            {
                id: 1,
                name: 'Admin',
                level: 100,
                description: 'Full system access with all permissions',
                active: true,
                metadata: JSON.stringify({ color: '#dc3545', icon: 'shield', bypassChecks: true }),
                created_at: new Date('2025-11-20 15:47:08'),
                updated_at: new Date('2025-11-20 15:47:08')
            },
            {
                id: 2,
                name: 'Manager',
                level: 50,
                description: 'Management access with most permissions except system administration',
                active: true,
                metadata: JSON.stringify({ color: '#ffc107', icon: 'briefcase' }),
                created_at: new Date('2025-11-20 15:47:08'),
                updated_at: new Date('2025-11-20 15:47:08')
            },
            {
                id: 3,
                name: 'Staff',
                level: 10,
                description: 'Standard staff access with limited permissions',
                active: true,
                metadata: JSON.stringify({ color: '#17a2b8', icon: 'user' }),
                created_at: new Date('2025-11-20 15:47:08'),
                updated_at: new Date('2025-11-20 15:47:08')
            },
            {
                id: 4,
                name: 'Doctor',
                level: 0,
                description: null,
                active: true,
                metadata: null,
                created_at: new Date('2025-11-20 15:47:08'),
                updated_at: new Date('2025-11-20 15:47:08')
            },
            {
                id: 5,
                name: 'Nurse',
                level: 10,
                description: '',
                active: true,
                metadata: JSON.stringify({}),
                created_at: new Date('2025-11-21 00:32:13'),
                updated_at: new Date('2025-11-21 00:32:13')
            }
        ];

        // Insert or update roles
        for (const role of roles) {
            const [existing] = await queryInterface.sequelize.query(
                `SELECT id FROM tbl_role WHERE id = ${role.id}`
            );

            if (existing.length > 0) {
                await queryInterface.sequelize.query(
                    `UPDATE tbl_role SET 
                        name = ${queryInterface.sequelize.escape(role.name)},
                        level = ${role.level},
                        description = ${role.description ? queryInterface.sequelize.escape(role.description) : 'NULL'},
                        active = ${role.active ? 1 : 0},
                        metadata = ${role.metadata ? queryInterface.sequelize.escape(role.metadata) : 'NULL'},
                        updated_at = ${queryInterface.sequelize.escape(role.updated_at)}
                    WHERE id = ${role.id}`
                );
            } else {
                await queryInterface.bulkInsert('tbl_role', [role]);
            }
        }
        console.log('   ✅ Seeded 5 roles');

        // 2. Seed Permissions (119 permissions from dump)
        // Resources: users, patients, doctors, appointments, sales, purchases, stock, reports, settings, dashboard, emr, inpatients, laboratory, pharmacy, warehouse, expenses, suppliers, stockExpense
        // Actions: create, read, update, delete, manage, export, import
        const resources = [
            'users', 'patients', 'doctors', 'appointments', 'sales', 'purchases',
            'stock', 'reports', 'settings', 'dashboard', 'emr', 'inpatients',
            'laboratory', 'pharmacy', 'warehouse', 'expenses', 'suppliers', 'stockExpense'
        ];
        const actions = ['create', 'read', 'update', 'delete', 'manage', 'export', 'import'];

        const permissions = [];
        let permissionId = 1;

        for (const resource of resources) {
            for (const action of actions) {
                permissions.push({
                    id: permissionId++,
                    resource,
                    action,
                    description: `${action.charAt(0).toUpperCase() + action.slice(1)} ${resource}`,
                    active: true,
                    metadata: JSON.stringify({ category: resource }),
                    created_at: new Date('2025-11-20 15:47:08'),
                    updated_at: new Date('2025-11-20 15:47:08')
                });
            }
        }

        // Insert or update permissions
        for (const permission of permissions) {
            const [existing] = await queryInterface.sequelize.query(
                `SELECT id FROM tbl_permission WHERE id = ${permission.id}`
            );

            if (existing.length > 0) {
                await queryInterface.sequelize.query(
                    `UPDATE tbl_permission SET 
                        resource = ${queryInterface.sequelize.escape(permission.resource)},
                        action = ${queryInterface.sequelize.escape(permission.action)},
                        description = ${queryInterface.sequelize.escape(permission.description)},
                        active = ${permission.active ? 1 : 0},
                        metadata = ${queryInterface.sequelize.escape(permission.metadata)},
                        updated_at = ${queryInterface.sequelize.escape(permission.updated_at)}
                    WHERE id = ${permission.id}`
                );
            } else {
                await queryInterface.bulkInsert('tbl_permission', [permission]);
            }
        }
        console.log(`   ✅ Seeded ${permissions.length} permissions`);

        // 3. Seed Role Permissions
        // Admin (role_id: 1) gets all permissions (1-119, with some disabled)
        // Manager (role_id: 2) gets most permissions
        // Staff (role_id: 3) gets limited permissions

        // Admin gets all permissions (granted: true for most, some disabled)
        const adminRolePermissions = [];
        for (let permId = 1; permId <= 119; permId++) {
            // Some permissions are disabled (64, 67, 68, 69, 70 based on dump)
            const granted = ![64, 67, 68, 69, 70].includes(permId);
            const active = granted;
            
            adminRolePermissions.push({
                role_id: 1,
                permission_id: permId,
                granted,
                active,
                metadata: JSON.stringify({}),
                created_at: new Date('2025-11-20 15:47:08'),
                updated_at: granted ? new Date('2025-11-20 15:47:08') : new Date('2025-11-21 00:31:52')
            });
        }

        // Manager gets permissions: 2, 8-19, 22-35, 36-49, 50-56, 58, 64-70, 71-119
        const managerPermissionIds = [
            2, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28,
            29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49,
            50, 51, 52, 53, 54, 55, 56, 58, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76,
            77, 78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96, 97,
            98, 99, 100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111, 112, 113, 114,
            115, 116, 117, 118, 119
        ];

        const managerRolePermissions = managerPermissionIds.map(permId => ({
            role_id: 2,
            permission_id: permId,
            granted: true,
            active: true,
            metadata: JSON.stringify({}),
            created_at: new Date('2025-11-20 15:47:08'),
            updated_at: new Date('2025-11-20 15:47:08')
        }));

        // Staff gets limited permissions
        const staffPermissionIds = [
            1, 2, 3, 6, 8, 9, 10, 13, 15, 16, 17, 20, 22, 23, 24, 27, 29, 30, 31, 34, 36, 37, 38,
            41, 43, 44, 45, 48, 50, 51, 52, 55, 57, 58, 59, 62, 64, 65, 66, 69, 71, 72, 73, 76,
            78, 79, 80, 83, 85, 86, 87, 90, 92, 93, 94, 97, 99, 100, 101, 104, 106, 107, 108,
            111, 113, 114, 115, 118
        ];

        const staffRolePermissions = staffPermissionIds.map(permId => ({
            role_id: 3,
            permission_id: permId,
            granted: true,
            active: true,
            metadata: JSON.stringify({}),
            created_at: new Date('2025-11-20 15:47:08'),
            updated_at: new Date('2025-11-20 15:47:08')
        }));

        // Insert or update role permissions
        const allRolePermissions = [...adminRolePermissions, ...managerRolePermissions, ...staffRolePermissions];

        for (const rp of allRolePermissions) {
            const [existing] = await queryInterface.sequelize.query(
                `SELECT id FROM tbl_role_permission WHERE role_id = ${rp.role_id} AND permission_id = ${rp.permission_id}`
            );

            if (existing.length > 0) {
                await queryInterface.sequelize.query(
                    `UPDATE tbl_role_permission SET 
                        granted = ${rp.granted ? 1 : 0},
                        active = ${rp.active ? 1 : 0},
                        metadata = ${queryInterface.sequelize.escape(rp.metadata)},
                        updated_at = ${queryInterface.sequelize.escape(rp.updated_at)}
                    WHERE role_id = ${rp.role_id} AND permission_id = ${rp.permission_id}`
                );
            } else {
                await queryInterface.bulkInsert('tbl_role_permission', [rp]);
            }
        }
        console.log(`   ✅ Seeded ${allRolePermissions.length} role-permission mappings`);

        // 4. Seed Default Admin User
        // Password hash from dump: $2a$10$1irptYeNDKFOXUNIFG4J.ulLfwCk49BnGZr/wsHIfZlwB0PT8w4ce
        // This is the hash for password "admin" (bcrypt)
        const adminUser = {
            id: 1,
            role_id: 1,
            name: 'admin',
            passwd: '$2a$10$1irptYeNDKFOXUNIFG4J.ulLfwCk49BnGZr/wsHIfZlwB0PT8w4ce', // password: admin
            ph_no: '09-123456789',
            nrc_no: '12/YGNXXX',
            address: 'Yangon',
            date: new Date('2025-10-12')
        };

        const [existingUser] = await queryInterface.sequelize.query(
            `SELECT id FROM tbl_user WHERE id = 1`
        );

        if (existingUser.length > 0) {
            await queryInterface.sequelize.query(
                `UPDATE tbl_user SET 
                    role_id = ${adminUser.role_id},
                    name = ${queryInterface.sequelize.escape(adminUser.name)},
                    passwd = ${queryInterface.sequelize.escape(adminUser.passwd)},
                    ph_no = ${queryInterface.sequelize.escape(adminUser.ph_no)},
                    nrc_no = ${queryInterface.sequelize.escape(adminUser.nrc_no)},
                    address = ${queryInterface.sequelize.escape(adminUser.address)},
                    date = ${queryInterface.sequelize.escape(adminUser.date)}
                WHERE id = 1`
            );
            console.log('   ✅ Updated admin user');
        } else {
            await queryInterface.bulkInsert('tbl_user', [adminUser]);
            console.log('   ✅ Created admin user (username: admin, password: admin)');
        }

        console.log('✅ Seeding completed!');
    },

    async down(queryInterface, Sequelize) {
        console.log('🗑️  Removing seeded data...');

        // Remove admin user
        await queryInterface.sequelize.query(
            `DELETE FROM tbl_user WHERE id = 1 AND name = 'admin'`
        );

        // Remove role permissions
        await queryInterface.sequelize.query(
            `DELETE FROM tbl_role_permission WHERE role_id IN (1, 2, 3)`
        );

        // Remove permissions
        await queryInterface.sequelize.query(
            `DELETE FROM tbl_permission WHERE id BETWEEN 1 AND 119`
        );

        // Remove roles (be careful - only if no users reference them)
        await queryInterface.sequelize.query(
            `DELETE FROM tbl_role WHERE id IN (1, 2, 3, 4, 5)`
        );

        console.log('✅ Rollback completed!');
    }
};

