/**
 * Seeder: Extract and Seed from SQL Dump File
 * Reads clinic_pro_db.sql and extracts INSERT statements for users, roles, permissions
 * 
 * This seeder directly uses the data from the SQL dump file
 */

'use strict';

const fs = require('fs');
const path = require('path');

module.exports = {
    async up(queryInterface, Sequelize) {
        const sqlFilePath = path.join(__dirname, '../../clinic_pro_db.sql');
        
        if (!fs.existsSync(sqlFilePath)) {
            console.log('⚠️  SQL dump file not found, skipping dump-based seeding');
            return;
        }

        console.log('📖 Reading SQL dump file...');
        const sqlContent = fs.readFileSync(sqlFilePath, 'utf8');

        // Extract INSERT statements
        const insertStatements = {
            permissions: sqlContent.match(/INSERT INTO `tbl_permission` VALUES[^;]+;/i)?.[0],
            roles: sqlContent.match(/INSERT INTO `tbl_role` VALUES[^;]+;/i)?.[0],
            rolePermissions: sqlContent.match(/INSERT INTO `tbl_role_permission` VALUES[^;]+;/i)?.[0],
            users: sqlContent.match(/INSERT INTO `tbl_user` VALUES[^;]+;/i)?.[0]
        };

        // Disable foreign key checks temporarily
        await queryInterface.sequelize.query('SET FOREIGN_KEY_CHECKS = 0');

        try {
            // 1. Seed Permissions
            if (insertStatements.permissions) {
                console.log('   📝 Seeding permissions...');
                // Clean the statement - INSERT INTO already has VALUES, just remove MySQL comments
                let cleanStatement = insertStatements.permissions
                    .replace(/\/\*!40000[^*]*\*\//g, '')
                    .trim();

                try {
                    // Use INSERT IGNORE to skip duplicates
                    cleanStatement = cleanStatement.replace(/INSERT INTO/i, 'INSERT IGNORE INTO');
                    await queryInterface.sequelize.query(cleanStatement);
                    console.log('   ✅ Permissions seeded');
                } catch (error) {
                    console.error('   ⚠️  Error seeding permissions:', error.message.substring(0, 200));
                }
            }

            // 2. Seed Roles
            if (insertStatements.roles) {
                console.log('   📝 Seeding roles...');
                let cleanStatement = insertStatements.roles
                    .replace(/\/\*!40000[^*]*\*\//g, '')
                    .trim();

                try {
                    cleanStatement = cleanStatement.replace(/INSERT INTO/i, 'INSERT IGNORE INTO');
                    await queryInterface.sequelize.query(cleanStatement);
                    console.log('   ✅ Roles seeded');
                } catch (error) {
                    console.error('   ⚠️  Error seeding roles:', error.message.substring(0, 200));
                }
            }

            // 3. Seed Role Permissions
            if (insertStatements.rolePermissions) {
                console.log('   📝 Seeding role permissions...');
                let cleanStatement = insertStatements.rolePermissions
                    .replace(/\/\*!40000[^*]*\*\//g, '')
                    .trim();

                try {
                    cleanStatement = cleanStatement.replace(/INSERT INTO/i, 'INSERT IGNORE INTO');
                    await queryInterface.sequelize.query(cleanStatement);
                    console.log('   ✅ Role permissions seeded');
                } catch (error) {
                    console.error('   ⚠️  Error seeding role permissions:', error.message.substring(0, 200));
                }
            }

            // 4. Seed Users
            if (insertStatements.users) {
                console.log('   📝 Seeding users...');
                let cleanStatement = insertStatements.users
                    .replace(/\/\*!40000[^*]*\*\//g, '')
                    .trim();

                try {
                    cleanStatement = cleanStatement.replace(/INSERT INTO/i, 'INSERT IGNORE INTO');
                    await queryInterface.sequelize.query(cleanStatement);
                    console.log('   ✅ Users seeded');
                    console.log('   📋 Default login: username=admin, password=admin');
                } catch (error) {
                    console.error('   ⚠️  Error seeding users:', error.message.substring(0, 200));
                }
            }

        } finally {
            // Re-enable foreign key checks
            await queryInterface.sequelize.query('SET FOREIGN_KEY_CHECKS = 1');
        }

        console.log('✅ Seeding from dump file completed!');
    },

    async down(queryInterface, Sequelize) {
        console.log('🗑️  Removing seeded data from dump...');

        await queryInterface.sequelize.query('SET FOREIGN_KEY_CHECKS = 0');

        try {
            // Remove in reverse order
            await queryInterface.sequelize.query(
                `DELETE FROM tbl_user WHERE id = 1`
            );
            await queryInterface.sequelize.query(
                `DELETE FROM tbl_role_permission WHERE role_id IN (1, 2, 3, 4, 5)`
            );
            await queryInterface.sequelize.query(
                `DELETE FROM tbl_permission WHERE id BETWEEN 1 AND 119`
            );
            await queryInterface.sequelize.query(
                `DELETE FROM tbl_role WHERE id IN (1, 2, 3, 4, 5)`
            );
        } finally {
            await queryInterface.sequelize.query('SET FOREIGN_KEY_CHECKS = 1');
        }

        console.log('✅ Rollback completed!');
    }
};

