/**
 * Clinic Pro V3 - Backend Server
 * Modern Web-Based Clinic & Pharmacy Management System
 */

const path = require('path');
require('dotenv').config({ path: path.join(__dirname, '.env') });
const express = require('express');
const cors = require('cors');
const helmet = require('helmet');
const compression = require('compression');
const morgan = require('morgan');
const rateLimit = require('express-rate-limit');
const session = require('express-session');

// Import middleware
const i18nextMiddleware = require('./middleware/i18n');
const errorHandler = require('./middleware/errorHandler');

// Import routes
const authRoutes = require('./routes/auth.routes');
const patientRoutes = require('./routes/patient.routes');
const inpatientRoutes = require('./routes/inpatient.routes');
const doctorRoutes = require('./routes/doctor.routes');
const appointmentRoutes = require('./routes/appointment.routes');
const stockRoutes = require('./routes/stock.routes');
const saleRoutes = require('./routes/sale.routes');
const purchaseRoutes = require('./routes/purchase.routes');
const reportRoutes = require('./routes/report.routes');
const settingsRoutes = require('./routes/settings.routes');
const dashboardRoutes = require('./routes/dashboard.routes');
const posRoutes = require('./routes/pos.routes');
const productRoutes = require('./routes/product.routes');
const serviceRoutes = require('./routes/service.routes');
const laboratoryItemRoutes = require('./routes/laboratory-item.routes');
const laboratoryPackageRoutes = require('./routes/laboratory-package.routes');
const emrRoutes = require('./routes/emr.routes');
const crudRoutes = require('./routes/crud.routes');
const vitalsRoutes = require('./routes/vitals.routes');
const notificationRoutes = require('./routes/notification.routes');
const userRoutes = require('./routes/user.routes');
const backupRoutes = require('./routes/backup.routes');
const supplierRoutes = require('./routes/supplier.routes');
const labRoutes = require('./routes/lab.routes'); // Legacy - for lab-reports page
const laboratoryRoutes = require('./routes/laboratory.routes');
const hospitalRoutes = require('./routes/hospital.routes');
const pharmacyRoutes = require('./routes/pharmacy.routes');
const analyticsRoutes = require('./routes/analytics.routes');
const healthRoutes = require('./routes/health.routes');
const depositRoutes = require('./routes/deposit.routes');
const uploadRoutes = require('./routes/upload.routes');
const invoiceRoutes = require('./routes/invoice.routes');
const returnRoutes = require('./routes/return.routes');
const warehouseRoutes = require('./routes/warehouse.routes');
const maintenanceRoutes = require('./routes/maintenance.routes');
const inpatientSalesRoutes = require('./routes/inpatient-sales.routes');
const deleteHistoryRoutes = require('./routes/delete-history.routes');
const dischargedPatientReportRoutes = require('./routes/discharged-patient-report.routes');
const inpatientDepositRoutes = require('./routes/inpatient-deposit.routes');
const inpatientReportRoutes = require('./routes/inpatient-report.routes');
const referringDoctorRoutes = require('./routes/referring-doctor.routes');
const expenseRoutes = require('./routes/expense.routes');
const permissionRoutes = require('./routes/permission.routes');

// Initialize Express app
const app = express();

// Trust proxy - only enable when actually behind a proxy
// Using `true` in development trips express-rate-limit safety checks.
const trustProxySetting = process.env.TRUST_PROXY
    ? process.env.TRUST_PROXY
    : (process.env.NODE_ENV === 'production' ? 1 : false);
app.set('trust proxy', trustProxySetting);

// Security middleware
app.use(helmet({
    contentSecurityPolicy: false,
    crossOriginEmbedderPolicy: false
}));

// Rate limiting - More generous limits for clinic management system
const limiter = rateLimit({
    windowMs: 15 * 60 * 1000, // 15 minutes
    max: 1000, // limit each IP to 1000 requests per windowMs (increased from 100)
    message: 'Too many requests from this IP, please try again later.',
    standardHeaders: true, // Return rate limit info in the `RateLimit-*` headers
    legacyHeaders: false, // Disable the `X-RateLimit-*` headers
    skip: (req) => {
        // Skip rate limiting for health checks and static files
        return req.path === '/api/health' || req.path.startsWith('/uploads/');
    }
});
app.use('/api/', limiter);

// CORS configuration
const allowedOrigins = [
    'http://localhost:3000',
    'http://localhost:4000',
    'http://localhost:5000',
    'https://demo.clinicpro.tech',
    'https://clinicpro.tech',
    'https://kaunghtet.clinicpro.tech',
    process.env.CORS_ORIGIN
].filter(Boolean); // Remove undefined values

app.use(cors({
    origin: function (origin, callback) {
        // Allow requests with no origin (like mobile apps or curl requests)
        if (!origin) return callback(null, true);
        
        // Check if origin is in allowed list
        if (allowedOrigins.indexOf(origin) !== -1) {
            return callback(null, true);
        }
        
        // Allow any subdomain of clinicpro.tech
        if (origin.match(/^https:\/\/[a-zA-Z0-9-]+\.clinicpro\.tech$/)) {
            return callback(null, true);
        }
        
        // Allow localhost and any IP address on common dev ports (local network access)
        if (origin.match(/^http:\/\/(localhost|127\.0\.0\.1|192\.168\.\d+\.\d+|10\.\d+\.\d+\.\d+):(3000|4000|5000)$/)) {
            return callback(null, true);
        }
        
            console.log('CORS blocked origin:', origin);
            callback(new Error('Not allowed by CORS'));
    },
    credentials: true,
    methods: ['GET', 'POST', 'PUT', 'DELETE', 'OPTIONS'],
    allowedHeaders: ['Content-Type', 'Authorization', 'X-Requested-With']
}));

// Static files - MUST BE FIRST for public access without authentication
const uploadsPath = path.join(__dirname, 'uploads');
console.log('📁 Serving uploads from:', uploadsPath);

// Debug middleware for uploads
app.use('/uploads', (req, res, next) => {
    console.log('📸 Upload request:', req.url);
    console.log('📸 Full path:', path.join(uploadsPath, req.url));
    console.log('📸 File exists:', require('fs').existsSync(path.join(uploadsPath, req.url)));
    next();
});

app.use('/uploads', express.static(uploadsPath, {
    setHeaders: (res, path) => {
        console.log('📸 Serving static file:', path);
        res.setHeader('Access-Control-Allow-Origin', '*');
        res.setHeader('Access-Control-Allow-Methods', 'GET');
        res.setHeader('Access-Control-Allow-Headers', 'Content-Type');
    }
}));

// Serve frontend static files
app.use(express.static(path.join(__dirname, '../frontend')));

// Body parser middleware
app.use(express.json({ limit: '50mb' }));
app.use(express.urlencoded({ extended: true, limit: '50mb' }));

// Session middleware (for permission caching)
app.use(session({
    secret: process.env.SESSION_SECRET || 'clinic-pro-session-secret-change-in-production',
    resave: false,
    saveUninitialized: false,
    cookie: {
        secure: process.env.NODE_ENV === 'production', // HTTPS only in production
        httpOnly: true,
        maxAge: 24 * 60 * 60 * 1000 // 24 hours
    },
    name: 'clinicpro.sid'
}));

// Compression middleware
app.use(compression());

// Logging middleware
if (process.env.NODE_ENV === 'development') {
    app.use(morgan('dev'));
} else {
    app.use(morgan('combined'));
}

// i18n middleware
app.use(i18nextMiddleware);

// Health check route
app.get('/health', (req, res) => {
    res.json({
        status: 'OK',
        timestamp: new Date().toISOString(),
        uptime: process.uptime(),
        environment: process.env.NODE_ENV
    });
});

// API routes
app.use('/api/auth', authRoutes);
app.use('/api/patients', patientRoutes);
app.use('/api/inpatients', inpatientRoutes);
app.use('/api/doctors', doctorRoutes);
app.use('/api/appointments', appointmentRoutes);
app.use('/api/stock', stockRoutes);
app.use('/api/sales', saleRoutes);
app.use('/api/purchases', purchaseRoutes);
app.use('/api/reports', reportRoutes);
app.use('/api/settings', settingsRoutes);
app.use('/api/dashboard', dashboardRoutes);
app.use('/api/pos', posRoutes);
app.use('/api/products', productRoutes);
app.use('/api/services', serviceRoutes);
app.use('/api/laboratory-items', laboratoryItemRoutes);
app.use('/api/laboratory-packages', laboratoryPackageRoutes);
app.use('/api/lab-orders', require('./routes/lab-order.routes'));
app.use('/api/emr', emrRoutes);
app.use('/api/crud', crudRoutes);
app.use('/api/vitals', vitalsRoutes);
app.use('/api/notifications', notificationRoutes);
app.use('/api/users', userRoutes);
app.use('/api/backup', backupRoutes);
app.use('/api/suppliers', supplierRoutes);
app.use('/api/lab', labRoutes); // Legacy - for lab-reports page
app.use('/api/laboratory', laboratoryRoutes);
app.use('/api/hospital', hospitalRoutes);
app.use('/api/pharmacy', pharmacyRoutes);
app.use('/api/analytics', analyticsRoutes);
app.use('/api/referring-doctors', referringDoctorRoutes);
app.use('/api/expenses', expenseRoutes);
app.use('/api/permissions', permissionRoutes);
app.use('/api/health', healthRoutes);
app.use('/api/deposits', depositRoutes);
app.use('/api/upload', uploadRoutes);
app.use('/api/invoice', invoiceRoutes);
app.use('/api/returns', returnRoutes);
app.use('/api/maintenance', maintenanceRoutes);
app.use('/api/warehouses', warehouseRoutes);
app.use('/api/inpatient-sales', inpatientSalesRoutes);
app.use('/api/delete-history', deleteHistoryRoutes);
app.use('/api/reports', dischargedPatientReportRoutes);
app.use('/api/inpatient-deposits', inpatientDepositRoutes);
app.use('/api/reports', inpatientReportRoutes);

// Translation endpoint
app.get('/api/translations/:lang', (req, res) => {
    const { lang } = req.params;
    const supportedLangs = ['en', 'my', 'zh'];
    
    if (!supportedLangs.includes(lang)) {
        return res.status(400).json({ error: 'Unsupported language' });
    }
    
    try {
        const translations = require(`./locales/${lang}/translation.json`);
        res.json(translations);
    } catch (error) {
        console.error('Translation loading error:', error);
        res.status(500).json({ error: 'Failed to load translations' });
    }
});

// API Welcome route
app.get('/api', (req, res) => {
    res.json({
        message: 'Welcome to Clinic Pro V3 API',
        version: '3.0.0',
        documentation: '/api/docs',
        endpoints: {
            auth: '/api/auth',
            pos: '/api/pos',
            patients: '/api/patients',
            doctors: '/api/doctors',
            emr: '/api/emr',
            products: '/api/products',
            dashboard: '/api/dashboard',
            settings: '/api/settings',
            reports: '/api/reports'
        }
    });
});

// Handle clean URLs for pages
app.get('/pages/:page', (req, res) => {
    const page = req.params.page;
    const filePath = path.join(__dirname, '../frontend/pages', `${page}.html`);
    
    // Check if the HTML file exists
    if (require('fs').existsSync(filePath)) {
        res.sendFile(filePath);
    } else {
        res.status(404).sendFile(path.join(__dirname, '../frontend/404.html'));
    }
});

// Serve frontend for all non-API routes (SPA fallback)
app.get('*', (req, res) => {
    // Don't serve HTML for API routes
    if (req.path.startsWith('/api/')) {
        return res.status(404).json({
            success: false,
            message: 'API route not found'
        });
    }
    
    // Serve index.html for root and other routes
    res.sendFile(path.join(__dirname, '../frontend/index.html'));
});

// Error handler middleware
app.use(errorHandler);

// Start server
const PORT = process.env.PORT || 5000;
const server = app.listen(PORT, () => {
    console.log(`
    ╔═══════════════════════════════════════════════════════╗
    ║                                                       ║
    ║         Clinic Pro V3 - Backend Server Running       ║
    ║                                                       ║
    ║   Environment: ${process.env.NODE_ENV || 'development'}${' '.repeat(36 - (process.env.NODE_ENV || 'development').length)}║
    ║   Port: ${PORT}${' '.repeat(46 - PORT.toString().length)}║
    ║   URL: http://localhost:${PORT}${' '.repeat(29 - PORT.toString().length)}║
    ║                                                       ║
    ╚═══════════════════════════════════════════════════════╝
    `);
});

// Handle unhandled promise rejections
process.on('unhandledRejection', (err) => {
    console.error('UNHANDLED REJECTION! 💥 Shutting down...');
    console.error(err);
    server.close(() => {
        process.exit(1);
    });
});

// Handle SIGTERM
process.on('SIGTERM', () => {
    console.log('👋 SIGTERM RECEIVED. Shutting down gracefully');
    server.close(() => {
        console.log('💥 Process terminated!');
    });
});

module.exports = app;

