const { query } = require('../config/database');

async function checkBedAvailability() {
    try {
        console.log('🏥 Checking bed availability...');

        // Get all beds
        const allBeds = await query('SELECT * FROM tbl_beds ORDER BY department_id, room_number, bed_number');
        console.log(`\n📊 Total beds in system: ${allBeds.length}`);

        // Group by status
        const statusCounts = {};
        allBeds.forEach(bed => {
            statusCounts[bed.status] = (statusCounts[bed.status] || 0) + 1;
        });

        console.log('\n📈 Bed status summary:');
        Object.entries(statusCounts).forEach(([status, count]) => {
            console.log(`  ${status}: ${count} beds`);
        });

        // Show available beds
        const availableBeds = allBeds.filter(bed => bed.status === 'Available');
        console.log(`\n✅ Available beds (${availableBeds.length}):`);
        availableBeds.forEach(bed => {
            console.log(`  - Bed ${bed.bed_number} in Room ${bed.room_number} (${bed.bed_type}) - ${bed.daily_rate} Ks/day`);
        });

        // Show occupied beds
        const occupiedBeds = allBeds.filter(bed => bed.status === 'Occupied');
        console.log(`\n🚫 Occupied beds (${occupiedBeds.length}):`);
        occupiedBeds.forEach(bed => {
            console.log(`  - Bed ${bed.bed_number} in Room ${bed.room_number} (${bed.bed_type}) - Patient: ${bed.current_patient_id || 'Unknown'}`);
        });

        // Show maintenance beds
        const maintenanceBeds = allBeds.filter(bed => bed.status === 'Maintenance');
        if (maintenanceBeds.length > 0) {
            console.log(`\n🔧 Maintenance beds (${maintenanceBeds.length}):`);
            maintenanceBeds.forEach(bed => {
                console.log(`  - Bed ${bed.bed_number} in Room ${bed.room_number} (${bed.bed_type})`);
            });
        }

        // Check for any beds with inconsistent data
        const inconsistentBeds = allBeds.filter(bed => 
            (bed.status === 'Available' && bed.current_patient_id) ||
            (bed.status === 'Occupied' && !bed.current_patient_id)
        );

        if (inconsistentBeds.length > 0) {
            console.log(`\n⚠️  Inconsistent bed data (${inconsistentBeds.length}):`);
            inconsistentBeds.forEach(bed => {
                console.log(`  - Bed ${bed.bed_number}: Status=${bed.status}, Patient=${bed.current_patient_id || 'None'}`);
            });
        }

        // Test bed selection for inpatient admission
        if (availableBeds.length > 0) {
            const testBed = availableBeds[0];
            console.log(`\n🧪 Testing bed selection for admission:`);
            console.log(`  Selected bed: ${testBed.bed_number} in Room ${testBed.room_number}`);
            console.log(`  Status: ${testBed.status}`);
            console.log(`  Current patient: ${testBed.current_patient_id || 'None'}`);
            console.log(`  Ready for admission: ${testBed.status === 'Available' && !testBed.current_patient_id ? 'YES' : 'NO'}`);
        } else {
            console.log(`\n❌ No available beds for admission!`);
        }

    } catch (error) {
        console.error('❌ Error checking bed availability:', error);
    } finally {
        console.log('\nBed availability check completed');
    }
}

checkBedAvailability();
