/**
 * Create All Missing Database Tables
 * Complete Hospital Management System Tables
 */

require('dotenv').config();
const { pool } = require('../config/database');

const createAllMissingTables = async () => {
    try {
        console.log('🏥 Creating Complete Hospital Management System Tables...');
        console.log('');

        // ========================================
        // LABORATORY MANAGEMENT TABLES
        // ========================================
        console.log('🧪 Creating Laboratory Management Tables...');

        const labTables = [
            // Lab Test Catalog
            `CREATE TABLE IF NOT EXISTS \`tbl_lab_test_catalog\` (
                \`id\` INT NOT NULL AUTO_INCREMENT,
                \`test_code\` VARCHAR(50) NOT NULL UNIQUE,
                \`test_name\` VARCHAR(255) NOT NULL,
                \`category\` VARCHAR(100) NOT NULL COMMENT 'Hematology, Chemistry, Microbiology, etc.',
                \`sample_type\` VARCHAR(50) NOT NULL COMMENT 'Blood, Urine, Stool, etc.',
                \`normal_range_male\` TEXT NULL,
                \`normal_range_female\` TEXT NULL,
                \`unit\` VARCHAR(50) NULL,
                \`method\` VARCHAR(100) NULL,
                \`department\` VARCHAR(100) NULL,
                \`price\` DECIMAL(10,2) DEFAULT 0,
                \`tat_hours\` INT DEFAULT 24 COMMENT 'Turnaround time in hours',
                \`critical_values\` TEXT NULL,
                \`preparation_required\` TEXT NULL,
                \`is_active\` TINYINT(1) DEFAULT 1,
                \`created_at\` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                \`updated_at\` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (\`id\`),
                INDEX \`idx_test_code\` (\`test_code\`),
                INDEX \`idx_category\` (\`category\`),
                INDEX \`idx_active\` (\`is_active\`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

            // Lab Order Sequence
            `CREATE TABLE IF NOT EXISTS \`tbl_lab_sequence\` (
                \`id\` INT NOT NULL AUTO_INCREMENT,
                \`last_order_number\` INT DEFAULT 0,
                \`year\` INT DEFAULT YEAR(CURDATE()),
                PRIMARY KEY (\`id\`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

            // Lab Orders
            `CREATE TABLE IF NOT EXISTS \`tbl_lab_orders\` (
                \`id\` INT NOT NULL AUTO_INCREMENT,
                \`order_number\` VARCHAR(50) NOT NULL UNIQUE,
                \`patient_id\` VARCHAR(50) NOT NULL,
                \`doctor_id\` INT NOT NULL,
                \`order_date\` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
                \`priority\` ENUM('Routine', 'Urgent', 'STAT') DEFAULT 'Routine',
                \`clinical_notes\` TEXT NULL,
                \`total_amount\` DECIMAL(10,2) DEFAULT 0,
                \`status\` ENUM('Pending', 'Collected', 'Processing', 'Completed', 'Cancelled') DEFAULT 'Pending',
                \`ordered_by\` INT NOT NULL,
                \`created_at\` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                \`updated_at\` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (\`id\`),
                INDEX \`idx_order_number\` (\`order_number\`),
                INDEX \`idx_patient\` (\`patient_id\`),
                INDEX \`idx_status\` (\`status\`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

            // Lab Order Details
            `CREATE TABLE IF NOT EXISTS \`tbl_lab_order_details\` (
                \`id\` INT NOT NULL AUTO_INCREMENT,
                \`order_id\` INT NOT NULL,
                \`test_id\` INT NOT NULL,
                \`test_name\` VARCHAR(255) NOT NULL,
                \`sample_collected\` TINYINT(1) DEFAULT 0,
                \`collection_date\` DATETIME NULL,
                \`collected_by\` INT NULL,
                \`status\` ENUM('Pending', 'Collected', 'Processing', 'Completed', 'Cancelled') DEFAULT 'Pending',
                \`price\` DECIMAL(10,2) DEFAULT 0,
                \`created_at\` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (\`id\`),
                INDEX \`idx_order\` (\`order_id\`),
                INDEX \`idx_test\` (\`test_id\`),
                FOREIGN KEY (\`order_id\`) REFERENCES \`tbl_lab_orders\`(\`id\`) ON DELETE CASCADE,
                FOREIGN KEY (\`test_id\`) REFERENCES \`tbl_lab_test_catalog\`(\`id\`) ON DELETE RESTRICT
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

            // Lab Results
            `CREATE TABLE IF NOT EXISTS \`tbl_lab_results\` (
                \`id\` INT NOT NULL AUTO_INCREMENT,
                \`order_detail_id\` INT NOT NULL,
                \`result_value\` TEXT NULL,
                \`result_unit\` VARCHAR(50) NULL,
                \`reference_range\` VARCHAR(255) NULL,
                \`flag\` ENUM('Normal', 'High', 'Low', 'Critical', 'Abnormal') DEFAULT 'Normal',
                \`comments\` TEXT NULL,
                \`result_date\` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
                \`verified_by\` INT NULL,
                \`verified_date\` DATETIME NULL,
                \`is_critical\` TINYINT(1) DEFAULT 0,
                \`critical_notified\` TINYINT(1) DEFAULT 0,
                \`created_at\` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                \`updated_at\` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (\`id\`),
                INDEX \`idx_order_detail\` (\`order_detail_id\`),
                INDEX \`idx_critical\` (\`is_critical\`),
                FOREIGN KEY (\`order_detail_id\`) REFERENCES \`tbl_lab_order_details\`(\`id\`) ON DELETE CASCADE
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`
        ];

        // Execute lab tables
        for (let i = 0; i < labTables.length; i++) {
            try {
                await pool.execute(labTables[i]);
                console.log(`✅ Lab table ${i + 1}/${labTables.length} created`);
            } catch (error) {
                console.log(`⚠️ Lab table ${i + 1} already exists or error: ${error.message}`);
            }
        }

        // ========================================
        // HOSPITAL OPERATIONS TABLES
        // ========================================
        console.log('\n🏥 Creating Hospital Operations Tables...');

        const hospitalTables = [
            // Departments
            `CREATE TABLE IF NOT EXISTS \`tbl_departments\` (
                \`id\` INT NOT NULL AUTO_INCREMENT,
                \`dept_code\` VARCHAR(20) NOT NULL UNIQUE,
                \`dept_name\` VARCHAR(255) NOT NULL,
                \`description\` TEXT NULL,
                \`head_of_department\` INT NULL,
                \`contact_number\` VARCHAR(20) NULL,
                \`location\` VARCHAR(255) NULL,
                \`bed_capacity\` INT DEFAULT 0,
                \`is_active\` TINYINT(1) DEFAULT 1,
                \`created_at\` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                \`updated_at\` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (\`id\`),
                INDEX \`idx_dept_code\` (\`dept_code\`),
                INDEX \`idx_active\` (\`is_active\`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

            // Beds
            `CREATE TABLE IF NOT EXISTS \`tbl_beds\` (
                \`id\` INT NOT NULL AUTO_INCREMENT,
                \`bed_number\` VARCHAR(20) NOT NULL,
                \`room_number\` VARCHAR(20) NOT NULL,
                \`department_id\` INT NOT NULL,
                \`bed_type\` ENUM('General', 'ICU', 'CCU', 'NICU', 'Private', 'Semi-Private') DEFAULT 'General',
                \`status\` ENUM('Available', 'Occupied', 'Cleaning', 'Maintenance', 'Reserved') DEFAULT 'Available',
                \`current_patient_id\` VARCHAR(50) NULL,
                \`admission_date\` DATETIME NULL,
                \`daily_rate\` DECIMAL(10,2) DEFAULT 0,
                \`equipment\` TEXT NULL COMMENT 'JSON array of equipment',
                \`notes\` TEXT NULL,
                \`is_active\` TINYINT(1) DEFAULT 1,
                \`created_at\` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                \`updated_at\` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (\`id\`),
                INDEX \`idx_bed_number\` (\`bed_number\`),
                INDEX \`idx_department\` (\`department_id\`),
                INDEX \`idx_status\` (\`status\`),
                FOREIGN KEY (\`department_id\`) REFERENCES \`tbl_departments\`(\`id\`) ON DELETE RESTRICT,
                UNIQUE KEY \`unique_bed\` (\`bed_number\`, \`room_number\`, \`department_id\`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='Bed Management - Added Cleaning status for automatic bed management after patient discharge'`,

            // OR Schedules
            `CREATE TABLE IF NOT EXISTS \`tbl_or_schedules\` (
                \`id\` INT NOT NULL AUTO_INCREMENT,
                \`or_number\` VARCHAR(20) NOT NULL,
                \`patient_id\` VARCHAR(50) NOT NULL,
                \`surgeon_id\` INT NOT NULL,
                \`anesthetist_id\` INT NULL,
                \`procedure_name\` VARCHAR(255) NOT NULL,
                \`procedure_code\` VARCHAR(50) NULL,
                \`scheduled_date\` DATETIME NOT NULL,
                \`estimated_duration\` INT NOT NULL COMMENT 'Duration in minutes',
                \`actual_start_time\` DATETIME NULL,
                \`actual_end_time\` DATETIME NULL,
                \`status\` ENUM('Scheduled', 'In_Progress', 'Completed', 'Cancelled', 'Postponed') DEFAULT 'Scheduled',
                \`priority\` ENUM('Routine', 'Urgent', 'Emergency') DEFAULT 'Routine',
                \`pre_op_notes\` TEXT NULL,
                \`post_op_notes\` TEXT NULL,
                \`complications\` TEXT NULL,
                \`created_by\` INT NOT NULL,
                \`created_at\` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                \`updated_at\` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (\`id\`),
                INDEX \`idx_or_number\` (\`or_number\`),
                INDEX \`idx_patient\` (\`patient_id\`),
                INDEX \`idx_surgeon\` (\`surgeon_id\`),
                INDEX \`idx_scheduled_date\` (\`scheduled_date\`),
                INDEX \`idx_status\` (\`status\`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

            // OR Equipment
            `CREATE TABLE IF NOT EXISTS \`tbl_or_equipment\` (
                \`id\` INT NOT NULL AUTO_INCREMENT,
                \`equipment_name\` VARCHAR(255) NOT NULL,
                \`equipment_code\` VARCHAR(50) NOT NULL UNIQUE,
                \`or_room\` VARCHAR(20) NOT NULL,
                \`status\` ENUM('Available', 'In_Use', 'Maintenance', 'Out_of_Service') DEFAULT 'Available',
                \`last_maintenance\` DATE NULL,
                \`next_maintenance\` DATE NULL,
                \`purchase_date\` DATE NULL,
                \`warranty_expiry\` DATE NULL,
                \`manufacturer\` VARCHAR(255) NULL,
                \`model\` VARCHAR(255) NULL,
                \`serial_number\` VARCHAR(255) NULL,
                \`notes\` TEXT NULL,
                \`is_active\` TINYINT(1) DEFAULT 1,
                \`created_at\` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                \`updated_at\` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (\`id\`),
                INDEX \`idx_equipment_code\` (\`equipment_code\`),
                INDEX \`idx_or_room\` (\`or_room\`),
                INDEX \`idx_status\` (\`status\`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`
        ];

        // Execute hospital tables
        for (let i = 0; i < hospitalTables.length; i++) {
            try {
                await pool.execute(hospitalTables[i]);
                console.log(`✅ Hospital table ${i + 1}/${hospitalTables.length} created`);
            } catch (error) {
                console.log(`⚠️ Hospital table ${i + 1} already exists or error: ${error.message}`);
            }
        }

        // ========================================
        // ANALYTICS DASHBOARD TABLES
        // ========================================
        console.log('\n📊 Creating Analytics Dashboard Tables...');

        const analyticsTables = [
            // Quality Metrics
            `CREATE TABLE IF NOT EXISTS \`tbl_quality_metrics\` (
                \`id\` INT NOT NULL AUTO_INCREMENT,
                \`metric_name\` VARCHAR(255) NOT NULL,
                \`metric_category\` VARCHAR(100) NOT NULL COMMENT 'Patient Safety, Clinical Effectiveness, etc.',
                \`numerator\` INT NOT NULL,
                \`denominator\` INT NOT NULL,
                \`percentage\` DECIMAL(5,2) NOT NULL,
                \`target_percentage\` DECIMAL(5,2) NULL,
                \`period_start\` DATE NOT NULL,
                \`period_end\` DATE NOT NULL,
                \`department_id\` INT NULL,
                \`data_source\` VARCHAR(255) NULL,
                \`benchmark_comparison\` VARCHAR(255) NULL,
                \`action_required\` TINYINT(1) DEFAULT 0,
                \`notes\` TEXT NULL,
                \`created_by\` INT NOT NULL,
                \`created_at\` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (\`id\`),
                INDEX \`idx_category\` (\`metric_category\`),
                INDEX \`idx_period\` (\`period_start\`, \`period_end\`),
                INDEX \`idx_department\` (\`department_id\`),
                FOREIGN KEY (\`department_id\`) REFERENCES \`tbl_departments\`(\`id\`) ON DELETE SET NULL
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

            // Clinical Indicators
            `CREATE TABLE IF NOT EXISTS \`tbl_clinical_indicators\` (
                \`id\` INT NOT NULL AUTO_INCREMENT,
                \`indicator_name\` VARCHAR(255) NOT NULL,
                \`indicator_type\` VARCHAR(100) NOT NULL,
                \`value\` DECIMAL(10,2) NOT NULL,
                \`unit\` VARCHAR(50) NULL,
                \`target_value\` DECIMAL(10,2) NULL,
                \`measurement_date\` DATE NOT NULL,
                \`department_id\` INT NULL,
                \`status\` ENUM('Above_Target', 'At_Target', 'Below_Target', 'Critical') NULL,
                \`trend\` ENUM('Improving', 'Stable', 'Declining') NULL,
                \`notes\` TEXT NULL,
                \`created_at\` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (\`id\`),
                INDEX \`idx_indicator_type\` (\`indicator_type\`),
                INDEX \`idx_measurement_date\` (\`measurement_date\`),
                INDEX \`idx_department\` (\`department_id\`),
                FOREIGN KEY (\`department_id\`) REFERENCES \`tbl_departments\`(\`id\`) ON DELETE SET NULL
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

            // Infection Surveillance
            `CREATE TABLE IF NOT EXISTS \`tbl_infection_surveillance\` (
                \`id\` INT NOT NULL AUTO_INCREMENT,
                \`patient_id\` VARCHAR(50) NOT NULL,
                \`infection_type\` VARCHAR(100) NOT NULL COMMENT 'HAI, SSI, UTI, CLABSI, etc.',
                \`infection_site\` VARCHAR(100) NOT NULL,
                \`onset_date\` DATE NOT NULL,
                \`detected_date\` DATE NOT NULL,
                \`days_since_admission\` INT NULL,
                \`risk_factors\` TEXT NULL,
                \`causative_organism\` VARCHAR(255) NULL,
                \`department_id\` INT NULL,
                \`associated_procedure\` VARCHAR(255) NULL,
                \`device_related\` TINYINT(1) DEFAULT 0,
                \`device_type\` VARCHAR(100) NULL,
                \`reported_by\` INT NOT NULL,
                \`status\` ENUM('Active', 'Resolved', 'Under_Treatment') DEFAULT 'Active',
                \`resolution_date\` DATE NULL,
                \`created_at\` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (\`id\`),
                INDEX \`idx_patient\` (\`patient_id\`),
                INDEX \`idx_infection_type\` (\`infection_type\`),
                INDEX \`idx_onset_date\` (\`onset_date\`),
                INDEX \`idx_department\` (\`department_id\`),
                FOREIGN KEY (\`department_id\`) REFERENCES \`tbl_departments\`(\`id\`) ON DELETE SET NULL
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

            // Readmissions
            `CREATE TABLE IF NOT EXISTS \`tbl_readmissions\` (
                \`id\` INT NOT NULL AUTO_INCREMENT,
                \`patient_id\` VARCHAR(50) NOT NULL,
                \`original_admission_date\` DATE NOT NULL,
                \`original_discharge_date\` DATE NOT NULL,
                \`readmission_date\` DATE NOT NULL,
                \`days_between\` INT NOT NULL,
                \`is_planned\` TINYINT(1) DEFAULT 0,
                \`original_diagnosis\` VARCHAR(255) NOT NULL,
                \`readmission_diagnosis\` VARCHAR(255) NOT NULL,
                \`is_related\` TINYINT(1) DEFAULT 0 COMMENT 'Related to original admission',
                \`department_id\` INT NULL,
                \`attending_physician\` INT NULL,
                \`risk_score\` DECIMAL(5,2) NULL,
                \`preventable\` ENUM('Yes', 'No', 'Possibly', 'Under_Review') NULL,
                \`notes\` TEXT NULL,
                \`created_at\` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (\`id\`),
                INDEX \`idx_patient\` (\`patient_id\`),
                INDEX \`idx_readmission_date\` (\`readmission_date\`),
                INDEX \`idx_days_between\` (\`days_between\`),
                INDEX \`idx_department\` (\`department_id\`),
                FOREIGN KEY (\`department_id\`) REFERENCES \`tbl_departments\`(\`id\`) ON DELETE SET NULL
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

            // Patient Satisfaction
            `CREATE TABLE IF NOT EXISTS \`tbl_patient_satisfaction\` (
                \`id\` INT NOT NULL AUTO_INCREMENT,
                \`patient_id\` VARCHAR(50) NOT NULL,
                \`survey_date\` DATE NOT NULL,
                \`survey_type\` VARCHAR(100) NOT NULL COMMENT 'Discharge, Outpatient, Emergency',
                \`overall_rating\` INT NOT NULL COMMENT '1-5 scale',
                \`staff_courtesy\` INT NULL COMMENT '1-5 scale',
                \`communication\` INT NULL COMMENT '1-5 scale',
                \`cleanliness\` INT NULL COMMENT '1-5 scale',
                \`pain_management\` INT NULL COMMENT '1-5 scale',
                \`discharge_information\` INT NULL COMMENT '1-5 scale',
                \`would_recommend\` TINYINT(1) NULL,
                \`department_id\` INT NULL,
                \`comments\` TEXT NULL,
                \`response_method\` ENUM('Phone', 'Email', 'Paper', 'Tablet') DEFAULT 'Paper',
                \`created_at\` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (\`id\`),
                INDEX \`idx_patient\` (\`patient_id\`),
                INDEX \`idx_survey_date\` (\`survey_date\`),
                INDEX \`idx_overall_rating\` (\`overall_rating\`),
                INDEX \`idx_department\` (\`department_id\`),
                FOREIGN KEY (\`department_id\`) REFERENCES \`tbl_departments\`(\`id\`) ON DELETE SET NULL
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

            // Operational Metrics
            `CREATE TABLE IF NOT EXISTS \`tbl_operational_metrics\` (
                \`id\` INT NOT NULL AUTO_INCREMENT,
                \`metric_name\` VARCHAR(255) NOT NULL,
                \`metric_type\` VARCHAR(100) NOT NULL COMMENT 'Efficiency, Utilization, Throughput',
                \`value\` DECIMAL(10,2) NOT NULL,
                \`unit\` VARCHAR(50) NULL,
                \`target_value\` DECIMAL(10,2) NULL,
                \`measurement_date\` DATE NOT NULL,
                \`department_id\` INT NULL,
                \`shift\` ENUM('Day', 'Evening', 'Night', 'All') NULL,
                \`data_points\` INT NULL COMMENT 'Number of observations',
                \`variance\` DECIMAL(10,2) NULL,
                \`notes\` TEXT NULL,
                \`created_at\` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (\`id\`),
                INDEX \`idx_metric_type\` (\`metric_type\`),
                INDEX \`idx_measurement_date\` (\`measurement_date\`),
                INDEX \`idx_department\` (\`department_id\`),
                FOREIGN KEY (\`department_id\`) REFERENCES \`tbl_departments\`(\`id\`) ON DELETE SET NULL
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

            // Incident Reports
            `CREATE TABLE IF NOT EXISTS \`tbl_incident_reports\` (
                \`id\` INT NOT NULL AUTO_INCREMENT,
                \`incident_number\` VARCHAR(50) NOT NULL UNIQUE,
                \`incident_date\` DATETIME NOT NULL,
                \`incident_type\` VARCHAR(100) NOT NULL COMMENT 'Patient Safety, Equipment, Staff, etc.',
                \`severity\` ENUM('Low', 'Medium', 'High', 'Critical') NOT NULL,
                \`location\` VARCHAR(255) NOT NULL,
                \`department_id\` INT NULL,
                \`patient_involved\` TINYINT(1) DEFAULT 0,
                \`patient_id\` VARCHAR(50) NULL,
                \`staff_involved\` TEXT NULL COMMENT 'JSON array of staff IDs',
                \`description\` TEXT NOT NULL,
                \`immediate_action\` TEXT NULL,
                \`investigation_required\` TINYINT(1) DEFAULT 0,
                \`investigation_findings\` TEXT NULL,
                \`corrective_actions\` TEXT NULL,
                \`preventive_measures\` TEXT NULL,
                \`status\` ENUM('Open', 'Under_Investigation', 'Closed', 'Referred') DEFAULT 'Open',
                \`reported_by\` INT NOT NULL,
                \`assigned_to\` INT NULL,
                \`target_completion\` DATE NULL,
                \`actual_completion\` DATE NULL,
                \`created_at\` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                \`updated_at\` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (\`id\`),
                INDEX \`idx_incident_number\` (\`incident_number\`),
                INDEX \`idx_incident_date\` (\`incident_date\`),
                INDEX \`idx_incident_type\` (\`incident_type\`),
                INDEX \`idx_severity\` (\`severity\`),
                INDEX \`idx_status\` (\`status\`),
                INDEX \`idx_department\` (\`department_id\`),
                FOREIGN KEY (\`department_id\`) REFERENCES \`tbl_departments\`(\`id\`) ON DELETE SET NULL
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

            // Incident Sequence
            `CREATE TABLE IF NOT EXISTS \`tbl_incident_sequence\` (
                \`id\` INT NOT NULL AUTO_INCREMENT,
                \`last_incident_number\` INT DEFAULT 0,
                \`year\` INT DEFAULT YEAR(CURDATE()),
                PRIMARY KEY (\`id\`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`
        ];

        // Execute analytics tables
        for (let i = 0; i < analyticsTables.length; i++) {
            try {
                await pool.execute(analyticsTables[i]);
                console.log(`✅ Analytics table ${i + 1}/${analyticsTables.length} created`);
            } catch (error) {
                console.log(`⚠️ Analytics table ${i + 1} already exists or error: ${error.message}`);
            }
        }

        // ========================================
        // INITIALIZE SEQUENCES
        // ========================================
        console.log('\n⚙️ Initializing Sequences...');

        // Initialize sequences if not exists
        const sequences = [
            { table: 'tbl_lab_sequence', column: 'last_order_number' },
            { table: 'tbl_incident_sequence', column: 'last_incident_number' }
        ];

        for (const seq of sequences) {
            try {
                await pool.execute(`INSERT INTO \`${seq.table}\` (\`${seq.column}\`, \`year\`) 
                    VALUES (0, YEAR(CURDATE())) 
                    ON DUPLICATE KEY UPDATE \`${seq.column}\` = \`${seq.column}\``);
                console.log(`✅ ${seq.table} initialized`);
            } catch (error) {
                console.log(`⚠️ ${seq.table} already exists`);
            }
        }

        console.log('\n🎉 ALL MISSING TABLES CREATED SUCCESSFULLY!');
        console.log('');
        console.log('📊 SUMMARY:');
        console.log(`   • Laboratory Management: ${labTables.length} tables`);
        console.log(`   • Hospital Operations: ${hospitalTables.length} tables`);
        console.log(`   • Analytics Dashboard: ${analyticsTables.length} tables`);
        console.log(`   • Pharmacy Management: Already complete`);
        console.log('');
        console.log('🚀 All modules are now ready for use!');
        
    } catch (error) {
        console.error('❌ Error creating tables:', error.message);
        throw error;
    } finally {
        await pool.end();
    }
};

// Run if called directly
if (require.main === module) {
    createAllMissingTables().catch(console.error);
}

module.exports = createAllMissingTables;
