/**
 * Create All Database Tables
 * Programmatic table creation for reliability
 */

require('dotenv').config();
const { pool } = require('../config/database');

const tables = [
    `CREATE TABLE IF NOT EXISTS tbl_alert (
        id INT NOT NULL AUTO_INCREMENT,
        cat_name VARCHAR(50) NULL,
        barcode VARCHAR(50) NULL,
        name VARCHAR(50) NULL,
        qty INT NULL,
        qty_alert INT NULL,
        date DATE NULL,
        status INT NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_cat_expenses (
        id INT NOT NULL AUTO_INCREMENT,
        name TEXT NOT NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_catalog (
        id INT NOT NULL AUTO_INCREMENT,
        name TEXT NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_catalog_2 (
        id INT NOT NULL AUTO_INCREMENT,
        name VARCHAR(50) NOT NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_appointment (
        id INT NOT NULL AUTO_INCREMENT,
        name VARCHAR(50) NULL,
        phone VARCHAR(50) NULL,
        case_ VARCHAR(50) NULL,
        dr_name VARCHAR(50) NULL,
        token_number VARCHAR(50) NULL,
        time DATETIME NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_customer (
        id INT NOT NULL AUTO_INCREMENT,
        name VARCHAR(50) NULL,
        ph_no VARCHAR(50) NULL,
        nrc_no VARCHAR(50) NULL,
        shop TEXT NULL,
        address TEXT NULL,
        date DATE NULL,
        remark TEXT NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_customer_debt (
        id INT NOT NULL AUTO_INCREMENT,
        customer_id INT NOT NULL,
        voucher_id INT NULL,
        debt DOUBLE NULL,
        pay DOUBLE NULL,
        remark TEXT NULL,
        date DATE NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_expenses (
        id INT NOT NULL AUTO_INCREMENT,
        cat_exp_id VARCHAR(50) NOT NULL,
        amount DOUBLE NOT NULL,
        user_name VARCHAR(50) NOT NULL,
        reason VARCHAR(50) NOT NULL,
        remark VARCHAR(50) NULL,
        payment_method VARCHAR(50) NULL,
        date DATE NOT NULL,
        status INT NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_form (
        id INT NOT NULL AUTO_INCREMENT,
        role_id VARCHAR(50) NOT NULL,
        sale VARCHAR(50) NOT NULL,
        buy VARCHAR(50) NOT NULL,
        product VARCHAR(50) NOT NULL,
        change_item_price VARCHAR(50) NOT NULL,
        cashier VARCHAR(50) NOT NULL,
        login_user VARCHAR(50) NOT NULL,
        customer VARCHAR(50) NOT NULL,
        supplier VARCHAR(50) NOT NULL,
        barcode VARCHAR(50) NOT NULL,
        expenses VARCHAR(50) NOT NULL,
        report VARCHAR(50) NOT NULL,
        setting VARCHAR(50) NOT NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_labour (
        id INT NOT NULL AUTO_INCREMENT,
        level_name VARCHAR(50) NULL,
        name VARCHAR(250) NULL,
        sex VARCHAR(10) NULL,
        age VARCHAR(50) NULL,
        nrc VARCHAR(250) NULL,
        ph_no VARCHAR(50) NULL,
        address VARCHAR(250) NULL,
        work_in DATE NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_labour_balance (
        id INT NOT NULL AUTO_INCREMENT,
        labour_name VARCHAR(50) NOT NULL,
        InMoney INT NULL,
        OutMoney INT NULL,
        About TEXT NULL,
        Remark TEXT NULL,
        date DATE NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_labour_level (
        id INT NOT NULL AUTO_INCREMENT,
        name VARCHAR(50) NOT NULL,
        salary VARCHAR(50) NOT NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_loss (
        id INT NOT NULL AUTO_INCREMENT,
        user_name VARCHAR(50) NULL,
        cat_id TEXT NULL,
        barcode VARCHAR(50) NULL,
        name VARCHAR(50) NULL,
        qty INT NULL,
        cost_price DOUBLE NULL,
        total_cost DOUBLE NULL,
        date DATE NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_purchase (
        id INT NOT NULL AUTO_INCREMENT,
        purchase_voucher_id INT NULL,
        cat_id VARCHAR(255) NULL,
        cat_id_2 VARCHAR(50) NULL,
        barcode VARCHAR(50) NULL,
        name TEXT NULL,
        qty INT NULL,
        qty_alert INT NULL,
        unit_name VARCHAR(50) NULL,
        cost_price INT NULL,
        total_cost_price INT NULL,
        sale_price_latli INT NULL,
        sale_price_latkar INT NULL,
        remark VARCHAR(255) NULL,
        date DATE NULL,
        status INT NULL,
        expire_date DATE NULL,
        alert_month INT NULL,
        alert_date DATE NULL,
        margin DOUBLE NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_purchase_return (
        id INT NOT NULL AUTO_INCREMENT,
        user_name VARCHAR(50) NULL,
        cat_id TEXT NULL,
        barcode VARCHAR(50) NULL,
        name VARCHAR(50) NULL,
        qty INT NULL,
        unit_name VARCHAR(50) NULL,
        cost_price DOUBLE NULL,
        total_cost DOUBLE NULL,
        date DATE NULL,
        status INT NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_purchase_voucher (
        id INT NOT NULL AUTO_INCREMENT,
        user_name VARCHAR(50) NULL,
        total_cost DOUBLE NOT NULL,
        supplier_name VARCHAR(50) NULL,
        Date DATE NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_role (
        id INT NOT NULL AUTO_INCREMENT,
        name VARCHAR(50) NOT NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_sale (
        id INT NOT NULL AUTO_INCREMENT,
        cat_id TEXT NULL,
        Barcode VARCHAR(50) NULL,
        Name TEXT NULL,
        SaleType VARCHAR(50) NULL,
        SalePrice DOUBLE NULL,
        dis DOUBLE NULL,
        Qty INT NULL,
        Total DOUBLE NULL,
        Profit DOUBLE NULL,
        VNo INT NULL,
        Date DATE NULL,
        Cashier VARCHAR(50) NULL,
        Refer VARCHAR(250) NULL,
        cost_price INT NULL,
        referring_doctor_id INT NULL,
        referring_doctor_name VARCHAR(100) NULL,
        warehouse_id INT NULL,
        PRIMARY KEY (id),
        INDEX idx_sale_vno (VNo),
        INDEX idx_sale_date (Date),
        INDEX idx_sale_warehouse_id (warehouse_id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_sale_draft (
        id INT NOT NULL AUTO_INCREMENT,
        cat_id TEXT NULL,
        Barcode VARCHAR(50) NULL,
        Name TEXT NULL,
        SaleType VARCHAR(50) NULL,
        SalePrice DOUBLE NULL,
        dis DOUBLE NULL,
        Qty INT NULL,
        Total DOUBLE NULL,
        Profit DOUBLE NULL,
        VNo INT NULL,
        Date DATE NULL,
        Cashier VARCHAR(50) NULL,
        Refer VARCHAR(250) NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_sale_draft_no (
        id INT NOT NULL,
        user_name VARCHAR(50) NULL,
        sub_total DOUBLE NULL,
        actual_cost DOUBLE NOT NULL,
        dis DOUBLE NULL,
        tax DOUBLE NULL,
        customer_name VARCHAR(50) NULL,
        Date DATETIME NULL,
        status INT NULL,
        payment VARCHAR(50) NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_sale_return (
        id INT NOT NULL AUTO_INCREMENT,
        cat_id TEXT NULL,
        barcode VARCHAR(50) NULL,
        name VARCHAR(50) NULL,
        sale_price DOUBLE NULL,
        qty INT NULL,
        total DOUBLE NULL,
        profit DOUBLE NULL,
        remark VARCHAR(50) NULL,
        date DATE NULL,
        voucher_return_id INT NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_sale_return_item (
        id INT NOT NULL AUTO_INCREMENT,
        cat_id TEXT NULL,
        barcode VARCHAR(50) NULL,
        name VARCHAR(50) NULL,
        sale_price DOUBLE NULL,
        qty INT NULL,
        total DOUBLE NULL,
        profit DOUBLE NULL,
        remark VARCHAR(50) NULL,
        date DATE NULL,
        vno INT NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_shopinfo (
        id INT NOT NULL AUTO_INCREMENT,
        name VARCHAR(50) NULL,
        type VARCHAR(50) NULL,
        ph_no VARCHAR(50) NULL,
        email VARCHAR(250) NULL,
        address VARCHAR(500) NULL,
        remark VARCHAR(250) NULL,
        thanks VARCHAR(250) NULL,
        photo LONGBLOB NULL,
        photo_path VARCHAR(255) NULL,
        user_code VARCHAR(250) NULL,
        product_code VARCHAR(250) NULL,
        software_code VARCHAR(250) NULL,
        printer_voucher_count INT NULL,
        printer VARCHAR(50) NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_stock (
        id INT NOT NULL AUTO_INCREMENT,
        cat_id VARCHAR(255) NULL,
        cat_id_2 VARCHAR(50) NULL,
        barcode VARCHAR(50) NULL,
        name TEXT NULL,
        qty INT NULL,
        qty_alert INT NULL,
        unit_name VARCHAR(50) NULL,
        cost_price INT NULL,
        sale_price_latli INT NULL,
        sale_price_latkar INT NULL,
        remark VARCHAR(255) NULL,
        expire_date DATE NULL,
        alert_month INT NULL,
        alert_date DATE NULL,
        margin DOUBLE NULL,
        image_path VARCHAR(255) NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_supplier (
        id INT NOT NULL AUTO_INCREMENT,
        name VARCHAR(50) NULL,
        ph_no VARCHAR(50) NULL,
        nrc_no VARCHAR(50) NULL,
        shop TEXT NULL,
        address TEXT NULL,
        date DATE NULL,
        remark TEXT NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_supplier_debt (
        id INT NOT NULL AUTO_INCREMENT,
        supplier_id INT NOT NULL,
        voucher_id INT NULL,
        debt DOUBLE NULL,
        pay DOUBLE NULL,
        remark TEXT NULL,
        date DATE NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_unit (
        id INT NOT NULL AUTO_INCREMENT,
        unit_name VARCHAR(50) NULL,
        remark VARCHAR(50) NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_service (
        id INT NOT NULL AUTO_INCREMENT,
        barcode VARCHAR(50) NOT NULL,
        service_name VARCHAR(50) NOT NULL,
        cat_id VARCHAR(50) NOT NULL,
        buy_price INT NOT NULL,
        sale_price INT NOT NULL,
        remark VARCHAR(50) NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_service_catalog (
        id INT NOT NULL AUTO_INCREMENT,
        name VARCHAR(50) NOT NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_payment_method (
        id INT NOT NULL AUTO_INCREMENT,
        name VARCHAR(50) NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_user_log (
        id INT NOT NULL AUTO_INCREMENT,
        user_name VARCHAR(50) NOT NULL,
        login_time DATETIME NULL,
        logout_time DATETIME NULL,
        total_sale DOUBLE NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_voucher (
        id INT NOT NULL,
        user_name VARCHAR(50) NULL,
        sub_total DOUBLE NULL,
        actual_cost DOUBLE NOT NULL,
        dis DOUBLE NULL,
        tax DOUBLE NULL,
        customer_name VARCHAR(50) NULL,
        Date DATETIME NULL,
        status INT NULL,
        payment VARCHAR(50) NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_patient (
        id INT NOT NULL AUTO_INCREMENT,
        name VARCHAR(50) NULL,
        age VARCHAR(50) NULL,
        phone VARCHAR(50) NULL,
        address VARCHAR(250) NULL,
        case_ TEXT NULL,
        remark TEXT NULL,
        date DATE NULL,
        dr_name VARCHAR(250) NULL,
        patient_id VARCHAR(250) NULL,
        room_no VARCHAR(250) NULL,
        deposit INT NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_sale_package_no (
        id INT NOT NULL,
        user_name VARCHAR(50) NULL,
        sub_total DOUBLE NULL,
        actual_cost DOUBLE NOT NULL,
        dis DOUBLE NULL,
        tax DOUBLE NULL,
        customer_name VARCHAR(50) NULL,
        Date DATETIME NULL,
        status INT NULL,
        payment VARCHAR(50) NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_sale_package (
        id INT NOT NULL AUTO_INCREMENT,
        cat_id TEXT NULL,
        Barcode VARCHAR(50) NULL,
        Name TEXT NULL,
        SaleType VARCHAR(50) NULL,
        SalePrice DOUBLE NULL,
        dis DOUBLE NULL,
        Qty INT NULL,
        Total DOUBLE NULL,
        Profit DOUBLE NULL,
        VNo INT NULL,
        Date DATE NULL,
        Cashier VARCHAR(50) NULL,
        Refer VARCHAR(250) NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_doctor (
        id INT NOT NULL AUTO_INCREMENT,
        name VARCHAR(50) NULL,
        phone VARCHAR(50) NULL,
        education VARCHAR(250) NULL,
        specialist VARCHAR(250) NULL,
        remark TEXT NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_patient_sale (
        id INT NOT NULL AUTO_INCREMENT,
        cat_id TEXT NULL,
        Barcode VARCHAR(50) NULL,
        Name TEXT NULL,
        SaleType VARCHAR(50) NULL,
        SalePrice DOUBLE NULL,
        dis DOUBLE NULL,
        Qty INT NULL,
        Total DOUBLE NULL,
        Profit DOUBLE NULL,
        VNo INT NULL,
        Date DATE NULL,
        Cashier VARCHAR(50) NULL,
        patient_id VARCHAR(250) NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_stock_expense (
        id INT NOT NULL AUTO_INCREMENT,
        user_name VARCHAR(50) NULL,
        cat_id TEXT NULL,
        barcode VARCHAR(50) NULL,
        name VARCHAR(250) NULL,
        qty INT NULL,
        unit VARCHAR(50) NULL,
        cost_price DOUBLE NULL,
        total_cost DOUBLE NULL,
        reason VARCHAR(250) NULL,
        date DATE NULL,
        warehouse_id INT NULL,
        PRIMARY KEY (id),
        INDEX idx_stock_expense_warehouse (warehouse_id),
        FOREIGN KEY (warehouse_id) REFERENCES tbl_warehouse(id) ON DELETE SET NULL ON UPDATE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_treatment (
        id INT NOT NULL AUTO_INCREMENT,
        patient_id VARCHAR(50) NULL,
        doctor_id INT NULL,
        chief_complaint TEXT NULL,
        present_illness TEXT NULL,
        Past_Diagnoses TEXT NULL,
        Surgeries_Procedures TEXT NULL,
        Hospitalizations TEXT NULL,
        Current_Medications TEXT NULL,
        Dosages TEXT NULL,
        Allergen TEXT NULL,
        Treatment_Type TEXT NULL,
        diagnosis TEXT NULL,
        prescription TEXT NULL,
        Date DATE NULL,
        Voucher_id VARCHAR(50) NULL,
        Recommendation TEXT NULL,
        vital_signs TEXT NULL,
        notes TEXT NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_case (
        id INT NOT NULL AUTO_INCREMENT,
        name VARCHAR(50) NOT NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_voucher_return (
        id INT NOT NULL AUTO_INCREMENT,
        user_name VARCHAR(50) NULL,
        sub_total DOUBLE NULL,
        actual_cost DOUBLE NOT NULL,
        dis DOUBLE NULL,
        tax DOUBLE NULL,
        customer_name VARCHAR(50) NULL,
        Date DATETIME NULL,
        status INT NULL,
        payment VARCHAR(50) NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_images (
        id INT NOT NULL AUTO_INCREMENT,
        patient_id VARCHAR(50) NULL,
        treatment_id INT NULL,
        image LONGBLOB NULL,
        image_path VARCHAR(255) NULL,
        upload_date DATETIME DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_sale_archive (
        id INT NOT NULL AUTO_INCREMENT,
        cat_id TEXT NULL,
        Barcode VARCHAR(50) NULL,
        Name TEXT NULL,
        SaleType VARCHAR(50) NULL,
        SalePrice DOUBLE NULL,
        dis DOUBLE NULL,
        Qty INT NULL,
        Total DOUBLE NULL,
        Profit DOUBLE NULL,
        VNo INT NULL,
        Date DATE NULL,
        Cashier VARCHAR(50) NULL,
        Refer VARCHAR(250) NULL,
        cost_price INT NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_voucher_archive (
        id INT NOT NULL,
        user_name VARCHAR(50) NULL,
        sub_total DOUBLE NULL,
        actual_cost DOUBLE NOT NULL,
        dis DOUBLE NULL,
        tax DOUBLE NULL,
        customer_name VARCHAR(50) NULL,
        Date DATETIME NULL,
        status INT NULL,
        payment VARCHAR(50) NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_inpatient (
        id INT NOT NULL AUTO_INCREMENT,
        name VARCHAR(50) NULL,
        age VARCHAR(50) NULL,
        phone VARCHAR(50) NULL,
        address VARCHAR(250) NULL,
        case_ TEXT NULL,
        remark TEXT NULL,
        start_date DATE NULL,
        dr_name VARCHAR(250) NULL,
        patient_id VARCHAR(250) NULL,
        room_no VARCHAR(250) NULL,
        deposit INT NULL,
        end_date DATE NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_inpatient_sale (
        id INT NOT NULL AUTO_INCREMENT,
        cat_id TEXT NULL,
        Barcode VARCHAR(50) NULL,
        Name TEXT NULL,
        SaleType VARCHAR(50) NULL,
        SalePrice DOUBLE NULL,
        dis DOUBLE NULL,
        Qty INT NULL,
        Total DOUBLE NULL,
        Profit DOUBLE NULL,
        VNo INT NULL,
        Date DATE NULL,
        Cashier VARCHAR(50) NULL,
        patient_id VARCHAR(250) NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_inpatient_saledebit (
        id INT NOT NULL AUTO_INCREMENT,
        patient_id VARCHAR(250) NOT NULL,
        voucher_id INT NULL,
        debt DOUBLE NULL,
        pay DOUBLE NULL,
        remark TEXT NULL,
        date DATE NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_user (
        id INT NOT NULL AUTO_INCREMENT,
        role_id VARCHAR(100) NOT NULL,
        name VARCHAR(100) NULL,
        passwd VARCHAR(100) NULL,
        ph_no VARCHAR(50) NULL,
        nrc_no VARCHAR(50) NULL,
        address VARCHAR(150) NULL,
        photo LONGBLOB NULL,
        date DATE NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    `CREATE TABLE IF NOT EXISTS tbl_township (
        id INT NOT NULL AUTO_INCREMENT,
        name VARCHAR(100) NULL,
        state VARCHAR(100) NULL,
        PRIMARY KEY (id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    // Laboratory Items - Lab test items that can be sold
    `CREATE TABLE IF NOT EXISTS tbl_laboratory_item (
        id INT NOT NULL AUTO_INCREMENT,
        barcode VARCHAR(50) NULL,
        item_name VARCHAR(50) NOT NULL,
        cat_id INT NULL,
        buy_price DECIMAL(10, 2) DEFAULT 0.00,
        sale_price DECIMAL(10, 2) DEFAULT 0.00,
        remark TEXT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        INDEX idx_barcode (barcode),
        INDEX idx_item_name (item_name),
        INDEX idx_cat_id (cat_id),
        FOREIGN KEY (cat_id) REFERENCES tbl_service_catalog(id) ON DELETE SET NULL
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='Laboratory test items that can be sold in POS and Inpatient Sales'`,

    // Laboratory Reports - Header table
    `CREATE TABLE IF NOT EXISTS tbl_lab_report (
        id INT NOT NULL AUTO_INCREMENT,
        report_no VARCHAR(50) NULL,
        patient_id VARCHAR(50) NOT NULL,
        doctor_id INT NULL,
        report_type VARCHAR(100) NULL,
        collected_at DATETIME NULL,
        reported_at DATETIME NULL,
        notes TEXT NULL,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        INDEX idx_lab_patient (patient_id),
        INDEX idx_lab_doctor (doctor_id),
        INDEX idx_lab_report_no (report_no)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    // Laboratory Report Items - Individual test results
    `CREATE TABLE IF NOT EXISTS tbl_lab_report_item (
        id INT NOT NULL AUTO_INCREMENT,
        report_id INT NOT NULL,
        test_name VARCHAR(150) NOT NULL,
        result_value VARCHAR(100) NULL,
        unit VARCHAR(50) NULL,
        reference_range VARCHAR(100) NULL,
        flag VARCHAR(10) NULL,
        remark VARCHAR(255) NULL,
        PRIMARY KEY (id),
        INDEX idx_lab_report_item_report (report_id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`,

    // Notification Read Status - Track which notifications users have read
    `CREATE TABLE IF NOT EXISTS tbl_notification_read (
        id INT NOT NULL AUTO_INCREMENT,
        notification_id VARCHAR(255) NOT NULL,
        user_id INT NULL,
        read_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        UNIQUE KEY unique_notification_user (notification_id, user_id),
        INDEX idx_user_id (user_id),
        INDEX idx_notification_id (notification_id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`
];

async function createAllTables() {
    console.log('🏗️  Creating All Database Tables...\n');
    
    const connection = await pool.getConnection();
    let successCount = 0;
    let errorCount = 0;

    for (const tableSql of tables) {
        try {
            await connection.query(tableSql);
            // Extract table name - handle both backtick and non-backtick formats
            // Match: CREATE TABLE [IF NOT EXISTS] `table_name` or CREATE TABLE [IF NOT EXISTS] table_name
            const match = tableSql.match(/CREATE TABLE\s+(?:IF NOT EXISTS\s+)?`?([a-zA-Z0-9_]+)`?/i);
            if (match && match[1] && match[1] !== 'IF') {
                console.log(`✅ ${match[1]}`);
                successCount++;
            } else {
                // Fallback: try to find table name before opening parenthesis
                const fallbackMatch = tableSql.match(/`([a-zA-Z0-9_]+)`\s*\(/);
                if (fallbackMatch) {
                    console.log(`✅ ${fallbackMatch[1]}`);
                    successCount++;
                }
            }
        } catch (error) {
            if (!error.message.includes('already exists')) {
                console.log(`❌ Error: ${error.message.substring(0, 60)}`);
                errorCount++;
            }
        }
    }

    // Create indexes
    console.log('\n📌 Creating Indexes...');
    const indexes = [
        'CREATE INDEX IF NOT EXISTS idx_patient_patient_id ON tbl_patient(patient_id)',
        'CREATE INDEX IF NOT EXISTS idx_sale_vno ON tbl_sale(VNo)',
        'CREATE INDEX IF NOT EXISTS idx_sale_date ON tbl_sale(Date)',
        'CREATE INDEX IF NOT EXISTS idx_stock_barcode ON tbl_stock(barcode)',
        'CREATE INDEX IF NOT EXISTS idx_treatment_patient ON tbl_treatment(patient_id)'
    ];

    for (const indexSql of indexes) {
        try {
            await connection.query(indexSql);
            const match = indexSql.match(/idx_(\w+)/);
            if (match) console.log(`✅ ${match[0]}`);
        } catch (error) {
            // Ignore
        }
    }

    // Add foreign key constraints (only if referenced tables exist)
    console.log('\n🔗 Adding Foreign Key Constraints...');
    const foreignKeys = [
        {
            sql: `ALTER TABLE tbl_sale 
                  ADD CONSTRAINT fk_sale_warehouse 
                  FOREIGN KEY (warehouse_id) 
                  REFERENCES tbl_warehouse(id) 
                  ON DELETE SET NULL 
                  ON UPDATE CASCADE`,
            name: 'fk_sale_warehouse'
        }
    ];

    for (const fk of foreignKeys) {
        try {
            // Check if the referenced table exists
            const [tables] = await connection.query(
                "SELECT COUNT(*) as count FROM information_schema.tables WHERE table_schema = DATABASE() AND table_name = ?",
                [fk.sql.includes('tbl_warehouse') ? 'tbl_warehouse' : '']
            );
            
            if (tables[0].count > 0) {
                // Check if constraint already exists
                const [constraints] = await connection.query(
                    `SELECT COUNT(*) as count 
                     FROM information_schema.table_constraints 
                     WHERE constraint_schema = DATABASE() 
                     AND constraint_name = ? 
                     AND table_name = 'tbl_sale'`,
                    [fk.name]
                );
                
                if (constraints[0].count === 0) {
                    await connection.query(fk.sql);
                    console.log(`✅ ${fk.name}`);
                } else {
                    console.log(`ℹ️  ${fk.name} already exists`);
                }
            } else {
                console.log(`⚠️  ${fk.name} skipped (referenced table doesn't exist yet)`);
            }
        } catch (error) {
            // Ignore constraint errors (might already exist or table doesn't exist)
            if (!error.message.includes('Duplicate key name') && !error.message.includes('already exists')) {
                console.log(`⚠️  ${fk.name}: ${error.message.substring(0, 50)}`);
            }
        }
    }

    connection.release();

    console.log(`\n🎉 Table Creation Complete!`);
    console.log(`✅ Success: ${successCount} tables`);
    console.log(`❌ Errors: ${errorCount}\n`);

    return successCount > 0;
}

if (require.main === module) {
    createAllTables().then(success => {
        process.exit(success ? 0 : 1);
    });
}

module.exports = { createAllTables };

