/**
 * Create Missing Hospital Tables
 * Surgery schedules, emergency admissions, operating rooms
 */

require('dotenv').config();
const { pool } = require('../config/database');

const createMissingHospitalTables = async () => {
    try {
        console.log('🏥 Creating Missing Hospital Tables...');

        // ========================================
        // OPERATING ROOMS TABLE
        // ========================================
        console.log('Creating operating rooms table...');

        const orRoomsTable = `CREATE TABLE IF NOT EXISTS \`tbl_operating_rooms\` (
            \`id\` INT NOT NULL AUTO_INCREMENT,
            \`or_code\` VARCHAR(20) NOT NULL UNIQUE,
            \`or_name\` VARCHAR(255) NOT NULL,
            \`department_id\` INT NOT NULL,
            \`capacity\` INT DEFAULT 1 COMMENT 'Number of operating tables',
            \`equipment\` TEXT NULL COMMENT 'JSON array of equipment',
            \`status\` ENUM('Available', 'In_Use', 'Maintenance', 'Cleaning') DEFAULT 'Available',
            \`is_active\` TINYINT(1) DEFAULT 1,
            \`notes\` TEXT NULL,
            \`created_at\` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            \`updated_at\` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (\`id\`),
            INDEX \`idx_or_code\` (\`or_code\`),
            INDEX \`idx_department\` (\`department_id\`),
            INDEX \`idx_status\` (\`status\`),
            FOREIGN KEY (\`department_id\`) REFERENCES \`tbl_departments\`(\`id\`) ON DELETE RESTRICT
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`;

        await pool.execute(orRoomsTable);
        console.log('✅ Operating rooms table created');

        // ========================================
        // SURGERY SCHEDULES TABLE
        // ========================================
        console.log('Creating surgery schedules table...');

        const surgerySchedulesTable = `CREATE TABLE IF NOT EXISTS \`tbl_surgery_schedules\` (
            \`id\` INT NOT NULL AUTO_INCREMENT,
            \`or_id\` INT NOT NULL,
            \`patient_id\` VARCHAR(50) NOT NULL,
            \`surgeon_id\` INT NOT NULL,
            \`anesthesiologist_id\` INT NULL,
            \`procedure_name\` VARCHAR(255) NOT NULL,
            \`procedure_code\` VARCHAR(50) NULL,
            \`scheduled_date\` DATE NOT NULL,
            \`scheduled_start_time\` TIME NOT NULL,
            \`scheduled_end_time\` TIME NULL,
            \`actual_start_time\` DATETIME NULL,
            \`actual_end_time\` DATETIME NULL,
            \`duration_minutes\` INT NULL,
            \`status\` ENUM('Scheduled', 'In_Progress', 'Completed', 'Cancelled', 'Postponed') DEFAULT 'Scheduled',
            \`priority\` ENUM('Routine', 'Urgent', 'Emergency') DEFAULT 'Routine',
            \`diagnosis\` TEXT NULL,
            \`pre_op_notes\` TEXT NULL,
            \`post_op_notes\` TEXT NULL,
            \`complications\` TEXT NULL,
            \`estimated_duration\` INT NOT NULL COMMENT 'Estimated duration in minutes',
            \`actual_duration\` INT NULL COMMENT 'Actual duration in minutes',
            \`created_by\` INT NOT NULL,
            \`created_at\` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            \`updated_at\` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (\`id\`),
            INDEX \`idx_or_id\` (\`or_id\`),
            INDEX \`idx_patient\` (\`patient_id\`),
            INDEX \`idx_surgeon\` (\`surgeon_id\`),
            INDEX \`idx_scheduled_date\` (\`scheduled_date\`),
            INDEX \`idx_status\` (\`status\`),
            FOREIGN KEY (\`or_id\`) REFERENCES \`tbl_operating_rooms\`(\`id\`) ON DELETE RESTRICT,
            FOREIGN KEY (\`surgeon_id\`) REFERENCES \`tbl_doctor\`(\`id\`) ON DELETE RESTRICT
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`;

        await pool.execute(surgerySchedulesTable);
        console.log('✅ Surgery schedules table created');

        // ========================================
        // EMERGENCY ADMISSIONS TABLE
        // ========================================
        console.log('Creating emergency admissions table...');

        const emergencyAdmissionsTable = `CREATE TABLE IF NOT EXISTS \`tbl_emergency_admissions\` (
            \`id\` INT NOT NULL AUTO_INCREMENT,
            \`patient_id\` VARCHAR(50) NOT NULL,
            \`arrival_date\` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            \`triage_level\` ENUM('Level_1', 'Level_2', 'Level_3', 'Level_4', 'Level_5') NOT NULL COMMENT '1=Critical, 5=Non-urgent',
            \`chief_complaint\` TEXT NOT NULL,
            \`vital_signs\` TEXT NULL COMMENT 'JSON of vital signs',
            \`assigned_doctor_id\` INT NULL,
            \`bed_assigned\` VARCHAR(20) NULL,
            \`disposition\` ENUM('Admitted', 'Discharged', 'Transferred', 'Expired') NULL,
            \`discharge_date\` DATETIME NULL,
            \`discharge_diagnosis\` TEXT NULL,
            \`notes\` TEXT NULL,
            \`created_by\` INT NOT NULL,
            \`created_at\` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            \`updated_at\` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (\`id\`),
            INDEX \`idx_patient\` (\`patient_id\`),
            INDEX \`idx_arrival_date\` (\`arrival_date\`),
            INDEX \`idx_triage_level\` (\`triage_level\`),
            INDEX \`idx_disposition\` (\`disposition\`),
            FOREIGN KEY (\`assigned_doctor_id\`) REFERENCES \`tbl_doctor\`(\`id\`) ON DELETE SET NULL
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci`;

        await pool.execute(emergencyAdmissionsTable);
        console.log('✅ Emergency admissions table created');

        // ========================================
        // UPDATE QUALITY METRICS TABLE - ADD meets_target COLUMN
        // ========================================
        console.log('Updating quality metrics table...');

        // Check if meets_target column exists
        const [columns] = await pool.execute(`
            SELECT COLUMN_NAME FROM INFORMATION_SCHEMA.COLUMNS
            WHERE TABLE_SCHEMA = 'clinic_pro_db'
            AND TABLE_NAME = 'tbl_quality_metrics'
            AND COLUMN_NAME = 'meets_target'
        `);

        if (columns.length === 0) {
            await pool.execute(`ALTER TABLE tbl_quality_metrics ADD COLUMN meets_target TINYINT(1) DEFAULT 0`);
            console.log('✅ Added meets_target column to quality_metrics table');
        } else {
            console.log('✅ meets_target column already exists');
        }

        // ========================================
        // SEED SAMPLE DATA
        // ========================================
        console.log('Seeding sample hospital data...');

        // Insert sample operating rooms
        const orRooms = [
            ['OR1', 'Operating Room 1', 7, 1, '["Anesthesia Machine", "Surgical Table", "C-Arm"]', 'Available'],
            ['OR2', 'Operating Room 2', 7, 1, '["Anesthesia Machine", "Surgical Table", "Electrosurgical Unit"]', 'Available'],
            ['OR3', 'Operating Room 3', 7, 1, '["Anesthesia Machine", "Surgical Table", "Ventilator"]', 'Available']
        ];

        for (const orRoom of orRooms) {
            try {
                await pool.execute(`
                    INSERT INTO tbl_operating_rooms (or_code, or_name, department_id, capacity, equipment, status)
                    VALUES (?, ?, ?, ?, ?, ?)
                    ON DUPLICATE KEY UPDATE or_name = VALUES(or_name)
                `, orRoom);
            } catch (error) {
                console.log(`⚠️ OR room ${orRoom[0]} already exists`);
            }
        }
        console.log(`✅ Added ${orRooms.length} operating rooms`);

        // Insert sample emergency departments
        const [emergencyDept] = await pool.execute(`
            SELECT id FROM tbl_departments WHERE dept_code = 'EMRG' LIMIT 1
        `);

        if (emergencyDept.length > 0) {
            const emergencyId = emergencyDept[0].id;

            // Insert sample emergency admission
            try {
                await pool.execute(`
                    INSERT INTO tbl_emergency_admissions
                    (patient_id, triage_level, chief_complaint, vital_signs, assigned_doctor_id, created_by)
                    VALUES (?, ?, ?, ?, ?, ?)
                `, ['P001', 'Level_2', 'Chest pain and shortness of breath', '{"blood_pressure": "140/90", "heart_rate": 88, "temperature": 98.6}', null, 1]);
                console.log('✅ Added sample emergency admission');
            } catch (error) {
                console.log('⚠️ Sample emergency admission already exists');
            }
        }

        console.log('\n🎉 All missing hospital tables created successfully!');
        console.log('');
        console.log('📊 HOSPITAL MODULE STATUS:');
        console.log('   ✅ Operating Rooms: Table and sample data');
        console.log('   ✅ Surgery Schedules: Table ready');
        console.log('   ✅ Emergency Admissions: Table ready');
        console.log('   ✅ Quality Metrics: meets_target column added');
        console.log('');
        console.log('🚀 Hospital Operations module is now fully functional!');

    } catch (error) {
        console.error('❌ Error creating hospital tables:', error.message);
        throw error;
    } finally {
        await pool.end();
    }
};

// Run if called directly
if (require.main === module) {
    createMissingHospitalTables().catch(console.error);
}

module.exports = createMissingHospitalTables;
