/**
 * Create tbl_inpatient_usage_settled table
 * Run this script to create the missing table
 */

const { query, testConnection } = require('../config/database');
const fs = require('fs');
const path = require('path');

async function createSettledTable() {
    try {
        console.log('🚀 Creating tbl_inpatient_usage_settled table...\n');
        
        // Test connection
        const connected = await testConnection();
        if (!connected) {
            throw new Error('Database connection failed');
        }
        
        // Read SQL file
        const sqlPath = path.join(__dirname, '../../database/migrations/009_create_inpatient_usage_settled_table.sql');
        const sql = fs.readFileSync(sqlPath, 'utf8');
        
        // Split by semicolons and execute each statement
        const statements = sql
            .split(';')
            .map(s => s.trim())
            .filter(s => s.length > 0 && !s.startsWith('--') && !s.startsWith('/*') && !s.startsWith('USE'));
        
        for (const statement of statements) {
            if (statement.trim()) {
                try {
                    await query(statement);
                    console.log('✅ Executed statement');
                } catch (error) {
                    // Ignore "already exists" errors
                    if (error.code === 'ER_TABLE_EXISTS_ERROR' || 
                        error.message.includes('already exists') ||
                        error.code === 'ER_DUP_KEYNAME') {
                        console.log('⚠️  Table or index already exists (skipping)');
                    } else {
                        throw error;
                    }
                }
            }
        }
        
        console.log('\n✅ Table tbl_inpatient_usage_settled created successfully!');
        process.exit(0);
    } catch (error) {
        console.error('\n❌ Error:', error.message);
        process.exit(1);
    }
}

// Run if called directly
if (require.main === module) {
    createSettledTable();
}

module.exports = { createSettledTable };

