/**
 * Create Patient Vitals Table
 */

const { query } = require('../config/database');

async function createVitalsTable() {
    try {
        console.log('🏥 Creating Patient Vitals Table...\n');

        // Create vitals table
        await query(`
            CREATE TABLE IF NOT EXISTS tbl_patient_vitals (
                id INT NOT NULL AUTO_INCREMENT,
                patient_id VARCHAR(50) NOT NULL,
                treatment_id INT NULL,
                blood_pressure_systolic INT NULL,
                blood_pressure_diastolic INT NULL,
                heart_rate INT NULL,
                temperature DECIMAL(4,1) NULL,
                respiratory_rate INT NULL,
                oxygen_saturation INT NULL,
                weight DECIMAL(5,2) NULL,
                height DECIMAL(5,2) NULL,
                bmi DECIMAL(4,2) NULL,
                blood_glucose DECIMAL(5,2) NULL,
                recorded_by VARCHAR(100) NULL,
                recorded_date DATETIME DEFAULT CURRENT_TIMESTAMP,
                notes TEXT NULL,
                PRIMARY KEY (id),
                INDEX idx_patient_vitals (patient_id),
                INDEX idx_vitals_date (recorded_date)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        `);
        
        console.log('✅ tbl_patient_vitals table created');

        // Seed sample vitals data
        console.log('\n🌱 Seeding sample vitals data...\n');
        
        await query(`
            INSERT IGNORE INTO tbl_patient_vitals 
            (patient_id, blood_pressure_systolic, blood_pressure_diastolic, heart_rate, temperature, respiratory_rate, oxygen_saturation, weight, height, bmi, blood_glucose, recorded_by, recorded_date, notes) 
            VALUES 
            -- OPD-002 (Hypertension patient) - 6 month history
            ('OPD-002', 150, 95, 82, 98.2, 18, 98, 75.5, 165, 27.7, NULL, 'Nurse May', DATE_SUB(NOW(), INTERVAL 180 DAY), 'Initial visit - High BP'),
            ('OPD-002', 145, 92, 80, 98.0, 17, 98, 75.2, 165, 27.6, NULL, 'Nurse May', DATE_SUB(NOW(), INTERVAL 150 DAY), 'Month 1 - Improving'),
            ('OPD-002', 142, 90, 78, 97.8, 18, 98, 74.8, 165, 27.5, NULL, 'Nurse May', DATE_SUB(NOW(), INTERVAL 120 DAY), 'Month 2 - Good response'),
            ('OPD-002', 138, 88, 76, 98.1, 17, 99, 74.5, 165, 27.4, NULL, 'Nurse May', DATE_SUB(NOW(), INTERVAL 90 DAY), 'Month 3 - Stable'),
            ('OPD-002', 135, 85, 75, 98.0, 18, 98, 74.0, 165, 27.2, NULL, 'Nurse May', DATE_SUB(NOW(), INTERVAL 60 DAY), 'Month 4 - Well controlled'),
            ('OPD-002', 132, 82, 74, 97.9, 17, 99, 73.5, 165, 27.0, NULL, 'Nurse May', DATE_SUB(NOW(), INTERVAL 30 DAY), 'Month 5 - Excellent'),
            ('OPD-002', 128, 80, 72, 98.0, 18, 99, 73.0, 165, 26.8, NULL, 'Nurse May', NOW(), 'Month 6 - Target achieved'),
            
            -- OPD-003 (Diabetes patient) - Blood glucose monitoring
            ('OPD-003', 135, 85, 76, 98.0, 18, 98, 82.0, 168, 29.1, 180, 'Nurse May', DATE_SUB(NOW(), INTERVAL 90 DAY), 'Initial - High glucose'),
            ('OPD-003', 132, 82, 75, 97.8, 17, 98, 80.5, 168, 28.5, 160, 'Nurse May', DATE_SUB(NOW(), INTERVAL 60 DAY), 'Improving glucose'),
            ('OPD-003', 130, 80, 74, 98.1, 18, 99, 79.0, 168, 28.0, 145, 'Nurse May', DATE_SUB(NOW(), INTERVAL 30 DAY), 'Good control'),
            ('OPD-003', 128, 78, 73, 98.0, 17, 99, 78.0, 168, 27.6, 125, 'Nurse May', NOW(), 'Target achieved'),
            
            -- IPD-001 (Pneumonia patient) - ICU monitoring
            ('IPD-001', 120, 75, 95, 101.2, 24, 92, 68.0, 172, 23.0, NULL, 'ICU Nurse', DATE_SUB(NOW(), INTERVAL 3 DAY), 'Admission - Fever, tachypnea'),
            ('IPD-001', 118, 74, 90, 100.5, 22, 94, 68.0, 172, 23.0, NULL, 'ICU Nurse', DATE_SUB(NOW(), INTERVAL 2 DAY), 'Day 2 - Improving'),
            ('IPD-001', 120, 76, 85, 99.2, 20, 96, 68.0, 172, 23.0, NULL, 'ICU Nurse', DATE_SUB(NOW(), INTERVAL 1 DAY), 'Day 3 - Much better'),
            ('IPD-001', 122, 78, 80, 98.4, 18, 98, 68.0, 172, 23.0, NULL, 'Nurse May', NOW(), 'Day 4 - Ready for discharge'),
            
            -- IPD-005 (Stroke patient) - Neuro monitoring
            ('IPD-005', 180, 100, 88, 98.0, 18, 97, 72.0, 170, 24.9, NULL, 'ICU Nurse', DATE_SUB(NOW(), INTERVAL 7 DAY), 'Admission - Hypertensive'),
            ('IPD-005', 165, 95, 85, 97.8, 17, 98, 72.0, 170, 24.9, NULL, 'ICU Nurse', DATE_SUB(NOW(), INTERVAL 6 DAY), 'Day 2 - BP lowering'),
            ('IPD-005', 155, 90, 82, 98.0, 18, 98, 72.0, 170, 24.9, NULL, 'Nurse May', DATE_SUB(NOW(), INTERVAL 5 DAY), 'Day 3 - Stable'),
            ('IPD-005', 145, 85, 80, 98.1, 17, 98, 72.0, 170, 24.9, NULL, 'Nurse May', DATE_SUB(NOW(), INTERVAL 3 DAY), 'Day 5 - Good response'),
            ('IPD-005', 138, 82, 78, 97.9, 18, 99, 72.0, 170, 24.9, NULL, 'Nurse May', NOW(), 'Day 8 - Target BP reached'),
            
            -- OPD-007 (Arthritis patient) - Weight management
            ('OPD-007', 140, 85, 75, 98.0, 17, 98, 88.0, 168, 31.2, NULL, 'Nurse May', DATE_SUB(NOW(), INTERVAL 120 DAY), 'Overweight - plan started'),
            ('OPD-007', 138, 84, 74, 98.1, 17, 98, 86.5, 168, 30.6, NULL, 'Nurse May', DATE_SUB(NOW(), INTERVAL 90 DAY), 'Lost 1.5kg'),
            ('OPD-007', 135, 82, 73, 97.9, 18, 99, 84.0, 168, 29.8, NULL, 'Nurse May', DATE_SUB(NOW(), INTERVAL 60 DAY), 'Lost 4kg total'),
            ('OPD-007', 132, 80, 72, 98.0, 17, 99, 82.0, 168, 29.1, NULL, 'Nurse May', DATE_SUB(NOW(), INTERVAL 30 DAY), 'Lost 6kg - excellent'),
            ('OPD-007', 130, 78, 71, 98.0, 18, 99, 80.5, 168, 28.5, NULL, 'Nurse May', NOW(), 'Lost 7.5kg - Continue'),
            
            -- IPD-003 (Dengue patient) - Fever monitoring
            ('IPD-003', 115, 70, 105, 103.5, 20, 98, 65.0, 170, 22.5, NULL, 'ICU Nurse', DATE_SUB(NOW(), INTERVAL 2 DAY), 'Admission - High fever'),
            ('IPD-003', 118, 72, 100, 102.0, 19, 98, 65.0, 170, 22.5, NULL, 'ICU Nurse', DATE_SUB(NOW(), INTERVAL 1 DAY), 'Day 2 - Fever persists'),
            ('IPD-003', 120, 75, 90, 99.5, 18, 98, 65.0, 170, 22.5, NULL, 'Nurse May', NOW(), 'Day 3 - Fever breaking'),
            
            -- Additional patients vitals
            ('OPD-014', 110, 70, 68, 98.2, 16, 99, 62.0, 160, 24.2, NULL, 'Nurse May', DATE_SUB(NOW(), INTERVAL 30 DAY), 'Pregnancy week 20'),
            ('OPD-014', 115, 72, 70, 98.0, 16, 99, 64.0, 160, 25.0, 90, 'Nurse May', NOW(), 'Pregnancy week 24 - All normal'),
            
            ('OPD-008', 125, 78, 72, 98.0, 18, 98, 58.0, 162, 22.1, NULL, 'Nurse May', DATE_SUB(NOW(), INTERVAL 15 DAY), 'Asthma checkup'),
            ('OPD-008', 122, 76, 70, 98.1, 17, 99, 58.0, 162, 22.1, NULL, 'Nurse May', NOW(), 'Asthma controlled')
        `);
        
        console.log('✅ Sample vitals data seeded (31 records)');
        console.log('\n🎉 Patient Vitals Table created and seeded successfully!\n');
        
        process.exit(0);
    } catch (error) {
        console.error('❌ Error:', error.message);
        process.exit(1);
    }
}

createVitalsTable();

