/**
 * Debug script to check warehouse stock and movements
 * Run: node backend/utils/debug-warehouse-stock.js
 */

const { query } = require('../config/database');

async function debugWarehouseStock() {
    try {
        console.log('🔍 Checking warehouse stock and movements...\n');
        
        // Check recent stock movements
        const movements = await query(`
            SELECT 
                sm.id,
                sm.movement_type,
                sm.warehouse_id,
                w.name as warehouse_name,
                sm.stock_id,
                s.name as stock_name,
                sm.quantity,
                sm.reference_type,
                sm.reference_id,
                sm.movement_date,
                sm.notes
            FROM tbl_stock_movement sm
            LEFT JOIN tbl_warehouse w ON sm.warehouse_id = w.id
            LEFT JOIN tbl_stock s ON sm.stock_id = s.id
            WHERE sm.reference_type = 'PURCHASE'
            ORDER BY sm.movement_date DESC
            LIMIT 10
        `);
        
        console.log('📦 Recent Purchase Movements:');
        console.table(movements);
        
        // Check warehouse stock
        const warehouseStock = await query(`
            SELECT 
                ws.warehouse_id,
                w.name as warehouse_name,
                ws.stock_id,
                s.name as stock_name,
                ws.quantity,
                ws.reserved_quantity,
                ws.last_updated
            FROM tbl_warehouse_stock ws
            LEFT JOIN tbl_warehouse w ON ws.warehouse_id = w.id
            LEFT JOIN tbl_stock s ON ws.stock_id = s.id
            ORDER BY ws.last_updated DESC
            LIMIT 10
        `);
        
        console.log('\n📊 Warehouse Stock:');
        console.table(warehouseStock);
        
        // Check for duplicate movements
        const duplicates = await query(`
            SELECT 
                warehouse_id,
                stock_id,
                reference_type,
                reference_id,
                COUNT(*) as count,
                SUM(quantity) as total_quantity
            FROM tbl_stock_movement
            WHERE reference_type = 'PURCHASE'
            GROUP BY warehouse_id, stock_id, reference_type, reference_id
            HAVING count > 1
        `);
        
        if (duplicates.length > 0) {
            console.log('\n⚠️  DUPLICATE MOVEMENTS FOUND:');
            console.table(duplicates);
        } else {
            console.log('\n✅ No duplicate movements found');
        }
        
        process.exit(0);
    } catch (error) {
        console.error('❌ Error:', error);
        process.exit(1);
    }
}

debugWarehouseStock();
