const { query } = require('../config/database');

/**
 * Record a delete event into tbl_delete_history.
 * @param {Object} params
 * @param {string} params.tableName - Source table name.
 * @param {string|number|null} params.recordId - Primary key of deleted row.
 * @param {Object|null} params.oldData - Deleted row snapshot (will be JSON stringified).
 * @param {number|null} params.deletedBy - User id (optional).
 * @param {string|null} params.username - Username (optional).
 * @param {string|null} params.reason - Optional reason/context.
 * @param {string|null} params.ip - Request IP (optional).
 * @param {string|null} params.userAgent - Request user agent (optional).
 */
async function recordDelete({
    tableName,
    recordId = null,
    oldData = null,
    deletedBy = null,
    username = null,
    reason = null,
    ip = null,
    userAgent = null
} = {}) {
    if (!tableName) return;

    try {
        await query(
            `INSERT INTO tbl_delete_history
                (table_name, record_id, deleted_by, username, old_data, reason, ip_address, user_agent)
             VALUES (?, ?, ?, ?, ?, ?, ?, ?)`,
            [
                tableName,
                recordId?.toString() ?? null,
                deletedBy ?? null,
                username ?? null,
                oldData ? JSON.stringify(oldData) : null,
                reason ?? null,
                ip ?? null,
                userAgent ?? null
            ]
        );
    } catch (err) {
        console.error('Failed to record delete history:', err.message);
    }
}

module.exports = {
    recordDelete,
};

