const { query } = require('../config/database');

async function fixDoctorTable() {
    try {
        console.log('🩺 Fixing doctor table structure...');

        // Add photo column to tbl_doctor if it doesn't exist
        await query(`
            ALTER TABLE tbl_doctor 
            ADD COLUMN IF NOT EXISTS photo VARCHAR(255) NULL COMMENT 'Doctor photo path'
        `).catch((error) => {
            if (error.code === 'ER_DUP_FIELDNAME') {
                console.log('⚠️  Photo column already exists in tbl_doctor');
            } else {
                console.error('❌ Error adding photo column:', error.message);
            }
        });

        // Add other useful columns if they don't exist
        await query(`
            ALTER TABLE tbl_doctor 
            ADD COLUMN IF NOT EXISTS email VARCHAR(100) NULL COMMENT 'Doctor email',
            ADD COLUMN IF NOT EXISTS address TEXT NULL COMMENT 'Doctor address',
            ADD COLUMN IF NOT EXISTS license_number VARCHAR(50) NULL COMMENT 'Medical license number',
            ADD COLUMN IF NOT EXISTS experience_years INT NULL COMMENT 'Years of experience',
            ADD COLUMN IF NOT EXISTS consultation_fee DECIMAL(10,2) DEFAULT 0.00 COMMENT 'Consultation fee',
            ADD COLUMN IF NOT EXISTS is_active TINYINT(1) DEFAULT 1 COMMENT 'Active status',
            ADD COLUMN IF NOT EXISTS created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            ADD COLUMN IF NOT EXISTS updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        `).catch((error) => {
            if (error.code === 'ER_DUP_FIELDNAME') {
                console.log('⚠️  Some columns already exist in tbl_doctor');
            } else {
                console.error('❌ Error adding additional columns:', error.message);
            }
        });

        // Add indexes for better performance
        await query(`
            ALTER TABLE tbl_doctor 
            ADD INDEX IF NOT EXISTS idx_doctor_active (is_active),
            ADD INDEX IF NOT EXISTS idx_doctor_specialist (specialist)
        `).catch((error) => {
            if (error.code === 'ER_DUP_KEYNAME') {
                console.log('⚠️  Some indexes already exist in tbl_doctor');
            } else {
                console.error('❌ Error adding indexes:', error.message);
            }
        });

        console.log('✅ Doctor table structure updated successfully!');
        
        // Verify the changes
        const verifySql = 'DESCRIBE tbl_doctor';
        const columns = await query(verifySql);
        console.log('\n📋 Updated tbl_doctor columns:');
        columns.forEach(col => {
            if (['photo', 'email', 'address', 'license_number', 'experience_years', 'consultation_fee', 'is_active'].includes(col.Field)) {
                console.log(`  ✅ ${col.Field}: ${col.Type} ${col.Null === 'YES' ? '(nullable)' : '(not null)'}`);
            }
        });

    } catch (error) {
        console.error('❌ Error fixing doctor table:', error);
    } finally {
        console.log('\nSetup completed');
    }
}

fixDoctorTable();
